<?php
/**
 * Plugin Name: UCO Auctions Navigation Patch
 * Description: Patches the uco-auctions plugin to fix prev/next lot navigation. Auto-reapplies after uco-auctions updates.
 * Version: 1.1.0
 * Author: SmoothByte
 * Requires Plugins: uco-auctions
 */

if (!defined('ABSPATH')) {
    exit;
}

class UCO_Navigation_Patch {
    
    const PATCH_DIR = WP_PLUGIN_DIR . '/uco-navigation-patch/patched-files/';
    const TARGET_DIR = WP_PLUGIN_DIR . '/uco-auctions/';
    const BACKUP_DIR = WP_CONTENT_DIR . '/uco-auctions-backup/';
    
    public function __construct() {
        // Hook into plugin update to re-apply patches
        add_action('upgrader_process_complete', array($this, 'reapply_after_update'), 10, 2);
        
        // Admin notice if patches need reapplication
        add_action('admin_notices', array($this, 'check_patch_status'));
    }
    
    /**
     * Re-apply patches after uco-auctions plugin is updated
     */
    public function reapply_after_update($upgrader_object, $options) {
        // Check if this is a plugin update
        if ($options['type'] !== 'plugin' || !isset($options['plugins'])) {
            return;
        }
        
        // Check if uco-auctions was updated
        foreach ($options['plugins'] as $plugin) {
            if (strpos($plugin, 'uco-auctions') !== false) {
                // UCO Auctions was updated, reapply patches
                $this->apply_patches();
                
                // Set a transient to show success message
                set_transient('uco_navigation_patch_reapplied', true, 60);
                break;
            }
        }
    }
    
    /**
     * Check if patches are applied and show admin notice
     */
    public function check_patch_status() {
        // Show success message after reapplication
        if (get_transient('uco_navigation_patch_reapplied')) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong>UCO Navigation Patch:</strong> Patches have been automatically re-applied after uco-auctions update.</p>
            </div>
            <?php
            delete_transient('uco_navigation_patch_reapplied');
        }
        
        // Check if patches need to be applied
        if (!$this->are_patches_applied()) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p><strong>UCO Navigation Patch:</strong> The navigation patches are not applied. <a href="<?php echo admin_url('plugins.php'); ?>">Try deactivating and reactivating this plugin.</a></p>
            </div>
            <?php
        }
    }
    
    /**
     * Check if patches are currently applied
     */
    private function are_patches_applied() {
        $test_file = self::TARGET_DIR . 'inc/helpers/LotHelper.php';
        if (!file_exists($test_file)) {
            return false;
        }
        
        $content = file_get_contents($test_file);
        // Check for our patch signature
        return (strpos($content, 'Get term_taxonomy_id from term_id since the SQL query needs term_taxonomy_id') !== false);
    }
    
    /**
     * Apply patches to files
     */
    private function apply_patches() {
        $files_to_patch = array(
            'inc/helpers/LotHelper.php',
            'templates/single-lot.php',
            'src/scss/_single.scss',
            'assets/css/libs.css',
            'assets/css/libs.min.css'
        );
        
        foreach ($files_to_patch as $file) {
            $source = self::PATCH_DIR . $file;
            $target = self::TARGET_DIR . $file;
            
            // Copy patched file
            if (file_exists($source)) {
                $target_dir = dirname($target);
                if (!file_exists($target_dir)) {
                    wp_mkdir_p($target_dir);
                }
                copy($source, $target);
            }
        }
    }
    
    public static function activate() {
        // Create backup directory if it doesn't exist
        if (!file_exists(self::BACKUP_DIR)) {
            wp_mkdir_p(self::BACKUP_DIR);
        }
        
        $files_to_patch = array(
            'inc/helpers/LotHelper.php',
            'templates/single-lot.php',
            'src/scss/_single.scss',
            'assets/css/libs.css',
            'assets/css/libs.min.css'
        );
        
        foreach ($files_to_patch as $file) {
            $source = self::PATCH_DIR . $file;
            $target = self::TARGET_DIR . $file;
            $backup = self::BACKUP_DIR . str_replace('/', '_', $file) . '.backup';
            
            // Create backup if file exists and backup doesn't exist
            if (file_exists($target) && !file_exists($backup)) {
                copy($target, $backup);
            }
            
            // Copy patched file
            if (file_exists($source)) {
                $target_dir = dirname($target);
                if (!file_exists($target_dir)) {
                    wp_mkdir_p($target_dir);
                }
                copy($source, $target);
            }
        }
        
        add_option('uco_navigation_patch_active', '1');
    }
    
    public static function deactivate() {
        // Restore original files from backup
        $files_to_restore = array(
            'inc/helpers/LotHelper.php',
            'templates/single-lot.php',
            'src/scss/_single.scss',
            'assets/css/libs.css',
            'assets/css/libs.min.css'
        );
        
        foreach ($files_to_restore as $file) {
            $backup = self::BACKUP_DIR . str_replace('/', '_', $file) . '.backup';
            $target = self::TARGET_DIR . $file;
            
            if (file_exists($backup)) {
                copy($backup, $target);
            }
        }
        
        delete_option('uco_navigation_patch_active');
    }
}

register_activation_hook(__FILE__, array('UCO_Navigation_Patch', 'activate'));
register_deactivation_hook(__FILE__, array('UCO_Navigation_Patch', 'deactivate'));

// Initialize the plugin
if (get_option('uco_navigation_patch_active')) {
    new UCO_Navigation_Patch();
}
