<?php
/**
 * Plugin Name: SmoothByte Security Pro
 * Plugin URI: https://smoothbyteit.dev
 * Description: Comprehensive WordPress security solution with enhanced malware scanning (deployment server integration), firewall protection, login security, security headers, and advanced hardening features. Built and maintained by SmoothByte IT.
 * Version: 1.5.4
 * Author: SmoothByte IT
 * Author URI: https://smoothbyteit.dev
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smoothbyte-security
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('SBSEC_VERSION', '1.5.4');
define('SBSEC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SBSEC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SBSEC_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main Security Plugin Class
 */
class SmoothByte_Security_Pro {

    /**
     * Single instance of the class
     */
    private static $instance = null;

    /**
     * Plugin options
     */
    private $options;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required dependencies
     */
    private function load_dependencies() {
        // Core security features
        require_once SBSEC_PLUGIN_DIR . 'includes/class-firewall.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-login-security.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-malware-scanner.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-security-hardening.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-file-monitor.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-security-logs.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-security-headers.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/class-auto-updater.php';
        
        // Admin interface
        if (is_admin()) {
            require_once SBSEC_PLUGIN_DIR . 'admin/class-admin-settings.php';
        }
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // Activation & Deactivation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Initialize components
        add_action('plugins_loaded', array($this, 'init'));
        
        // Add settings link on plugins page
        add_filter('plugin_action_links_' . SBSEC_PLUGIN_BASENAME, array($this, 'add_settings_link'));
    }

    /**
     * Initialize plugin components
     */
    public function init() {
        // Load options
        $this->options = get_option('sbsec_options', $this->get_default_options());

        // Initialize firewall
        if ($this->is_enabled('firewall')) {
            new SBSEC_Firewall($this->options);
        }

        // Initialize login security
        if ($this->is_enabled('login_security')) {
            new SBSEC_Login_Security($this->options);
        }

        // Initialize malware scanner
        if ($this->is_enabled('malware_scanner')) {
            new SBSEC_Malware_Scanner($this->options);
        }

        // Initialize security hardening
        if ($this->is_enabled('security_hardening')) {
            new SBSEC_Security_Hardening($this->options);
        }

        // Initialize file monitor
        if ($this->is_enabled('file_monitor')) {
            new SBSEC_File_Monitor($this->options);
        }

        // Initialize security logs
        new SBSEC_Security_Logs($this->options);
        
        // Initialize security headers
        if ($this->is_enabled('security_headers')) {
            new SBSEC_Security_Headers($this->options);
        }
        
        // Initialize auto-updater
        new SBSEC_Auto_Updater(SBSEC_PLUGIN_BASENAME, SBSEC_VERSION);

        // Initialize admin panel
        if (is_admin()) {
            new SBSEC_Admin_Settings($this->options);
        }
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        if (!get_option('sbsec_options')) {
            add_option('sbsec_options', $this->get_default_options());
        }

        // Create security logs table
        $this->create_tables();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled tasks
        wp_clear_scheduled_hook('sbsec_daily_scan');
        wp_clear_scheduled_hook('sbsec_hourly_check');
        wp_clear_scheduled_hook('sbsec_report_health');
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Create database tables
     */
    private function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Security logs table
        $table_name = $wpdb->prefix . 'sbsec_logs';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            event_type varchar(50) NOT NULL,
            severity varchar(20) NOT NULL,
            ip_address varchar(100) DEFAULT '' NOT NULL,
            user_id bigint(20) DEFAULT 0,
            message text NOT NULL,
            data longtext,
            PRIMARY KEY (id),
            KEY event_type (event_type),
            KEY timestamp (timestamp),
            KEY ip_address (ip_address)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Blocked IPs table
        $table_name = $wpdb->prefix . 'sbsec_blocked_ips';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ip_address varchar(100) NOT NULL,
            reason varchar(255) NOT NULL,
            blocked_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            expires_at datetime DEFAULT NULL,
            block_type varchar(20) DEFAULT 'manual' NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address),
            KEY expires_at (expires_at)
        ) $charset_collate;";
        
        dbDelta($sql);
        
        // File checksums table
        $table_name = $wpdb->prefix . 'sbsec_file_checksums';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            file_path varchar(500) NOT NULL,
            checksum varchar(64) NOT NULL,
            file_size bigint(20) NOT NULL,
            last_modified datetime NOT NULL,
            last_checked datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY file_path (file_path),
            KEY last_checked (last_checked)
        ) $charset_collate;";
        
        dbDelta($sql);
        
        // Scan results table
        $table_name = $wpdb->prefix . 'sbsec_scan_results';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            scan_time datetime NOT NULL,
            files_scanned int(11) DEFAULT 0,
            threats_found int(11) DEFAULT 0,
            warnings_found int(11) DEFAULT 0,
            scan_duration float DEFAULT 0,
            scan_data longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY (id),
            KEY scan_time (scan_time),
            KEY threats_found (threats_found)
        ) $charset_collate;";
        
        dbDelta($sql);
        
        // Quarantine table
        $table_name = $wpdb->prefix . 'sbsec_quarantine';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            original_path varchar(500) NOT NULL,
            quarantine_path varchar(500) NOT NULL,
            file_size bigint(20) DEFAULT 0,
            file_hash varchar(64) NOT NULL,
            quarantined_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            reason text,
            PRIMARY KEY (id),
            KEY original_path (original_path),
            KEY quarantined_at (quarantined_at)
        ) $charset_collate;";
        
        dbDelta($sql);
    }

    /**
     * Get default plugin options
     */
    private function get_default_options() {
        return array(
            // General Settings
            'enabled' => true,
            
            // Firewall Settings
            'firewall' => true,
            'firewall_block_bad_requests' => true,
            'firewall_rate_limiting' => true,
            'firewall_rate_limit' => 60, // requests per minute
            'firewall_geo_blocking' => false,
            'firewall_blocked_countries' => array(),
            'firewall_whitelist_ips' => array(),
            'firewall_blacklist_ips' => array(),
            
            // Login Security
            'login_security' => true,
            'login_limit_attempts' => true,
            'login_max_attempts' => 5,
            'login_lockout_duration' => 30, // minutes
            'login_2fa_enabled' => false,
            'login_captcha_enabled' => false,
            'login_hide_login_page' => false,
            'login_custom_url' => '',
            
            // Malware Scanner
            'malware_scanner' => true,
            'scan_schedule' => 'daily',
            'scan_wp_core' => true,
            'scan_plugins' => true,
            'scan_themes' => true,
            'scan_uploads' => true,
            'scan_suspicious_patterns' => true,
            'scan_email_notifications' => true,
            'scan_notification_email' => get_option('admin_email'),
            
            // Security Hardening
            'security_hardening' => true,
            'hardening_disable_file_editing' => true,
            'hardening_disable_xmlrpc' => true,
            'hardening_remove_wp_version' => true,
            'hardening_disable_directory_browsing' => true,
            'hardening_force_ssl_admin' => false,
            'hardening_disable_user_enumeration' => true,
            'hardening_protect_config' => true,
            
            // File Monitor
            'file_monitor' => true,
            'monitor_core_files' => true,
            'monitor_plugin_files' => true,
            'monitor_theme_files' => true,
            'monitor_alert_on_change' => true,
            
            // Security Headers
            'security_headers' => true,
            'security_headers_xframe' => true,
            'security_headers_xframe_mode' => 'SAMEORIGIN',
            'security_headers_xss' => true,
            'security_headers_xss_mode' => 'block',
            'security_headers_content_type' => true,
            'security_headers_hsts' => false, // Disabled by default (requires SSL)
            'security_headers_hsts_max_age' => 31536000, // 1 year
            'security_headers_hsts_subdomains' => false,
            'security_headers_hsts_preload' => false,
            'security_headers_referrer' => true,
            'security_headers_referrer_policy' => 'strict-origin-when-cross-origin',
            'security_headers_permissions' => true,
            'security_headers_permissions_geolocation' => 'none',
            'security_headers_permissions_microphone' => 'none',
            'security_headers_permissions_camera' => 'none',
            'security_headers_permissions_payment' => 'none',
            'security_headers_permissions_usb' => 'none',
            
            // Notifications
            'email_notifications' => true,
            'notification_email' => get_option('admin_email'),
            
            // Advanced
            'debug_mode' => false,
            'log_retention_days' => 30,
        );
    }

    /**
     * Check if a feature is enabled
     */
    private function is_enabled($feature) {
        return !empty($this->options[$feature]);
    }

    /**
     * Add settings link on plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="admin.php?page=smoothbyte-security">' . __('Settings', 'smoothbyte-security') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

// Initialize the plugin
function smoothbyte_security_pro() {
    return SmoothByte_Security_Pro::get_instance();
}

// Start the plugin
smoothbyte_security_pro();
