<?php
/**
 * Referrer-Policy Header
 * Controls how much referrer information is sent with requests
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-base-header.php';

/**
 * Referrer-Policy Header Implementation
 */
class SBSEC_Referrer_Policy extends SBSEC_Base_Header {
    
    /**
     * Constructor
     */
    public function __construct($options) {
        parent::__construct($options);
        $this->header_name = 'Referrer-Policy';
    }
    
    /**
     * Check if header is enabled
     */
    public function is_enabled() {
        return !empty($this->options['security_headers_referrer']);
    }
    
    /**
     * Get header name
     */
    public function get_name() {
        return 'Referrer-Policy';
    }
    
    /**
     * Get header description
     */
    public function get_description() {
        return 'Controls how much referrer information (sent via the Referer header) should be included with requests.';
    }
    
    /**
     * Get header value
     */
    protected function get_value() {
        $policy = isset($this->options['security_headers_referrer_policy']) 
            ? $this->options['security_headers_referrer_policy'] 
            : 'strict-origin-when-cross-origin';
        
        $valid_policies = array_keys($this->get_policies());
        
        if (!in_array($policy, $valid_policies)) {
            $policy = 'strict-origin-when-cross-origin';
        }
        
        return $policy;
    }
    
    /**
     * Get available policies
     */
    public function get_policies() {
        return array(
            'no-referrer' => array(
                'label' => 'no-referrer',
                'description' => 'Never send referrer information.',
            ),
            'no-referrer-when-downgrade' => array(
                'label' => 'no-referrer-when-downgrade',
                'description' => 'Send referrer to same protocol level (HTTPS→HTTPS), but not to less secure (HTTPS→HTTP).',
            ),
            'origin' => array(
                'label' => 'origin',
                'description' => 'Send only origin (e.g., https://example.com) without full URL.',
            ),
            'origin-when-cross-origin' => array(
                'label' => 'origin-when-cross-origin',
                'description' => 'Send full URL for same-origin requests, only origin for cross-origin.',
            ),
            'same-origin' => array(
                'label' => 'same-origin',
                'description' => 'Send referrer only for same-origin requests.',
            ),
            'strict-origin' => array(
                'label' => 'strict-origin',
                'description' => 'Send origin only when protocol security level stays same (HTTPS→HTTPS).',
            ),
            'strict-origin-when-cross-origin' => array(
                'label' => 'strict-origin-when-cross-origin (Recommended)',
                'description' => 'Send full URL for same-origin, origin for cross-origin HTTPS, nothing for HTTP.',
            ),
            'unsafe-url' => array(
                'label' => 'unsafe-url',
                'description' => 'Always send full URL (not recommended).',
            ),
        );
    }
}
