<?php
/**
 * Permissions-Policy Header (formerly Feature-Policy)
 * Controls which browser features can be used
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-base-header.php';

/**
 * Permissions-Policy Header Implementation
 */
class SBSEC_Permissions_Policy extends SBSEC_Base_Header {
    
    /**
     * Constructor
     */
    public function __construct($options) {
        parent::__construct($options);
        $this->header_name = 'Permissions-Policy';
    }
    
    /**
     * Check if header is enabled
     */
    public function is_enabled() {
        return !empty($this->options['security_headers_permissions']);
    }
    
    /**
     * Get header name
     */
    public function get_name() {
        return 'Permissions-Policy';
    }
    
    /**
     * Get header description
     */
    public function get_description() {
        return 'Controls which browser features and APIs can be used on your site (e.g., geolocation, camera, microphone).';
    }
    
    /**
     * Get header value
     */
    protected function get_value() {
        $features = $this->get_default_features();
        $policies = array();
        
        foreach ($features as $feature => $default) {
            $option_key = 'security_headers_permissions_' . $feature;
            $value = isset($this->options[$option_key]) ? $this->options[$option_key] : $default;
            
            if ($value === 'none') {
                $policies[] = $feature . '=()';
            } elseif ($value === 'self') {
                $policies[] = $feature . '=(self)';
            } elseif ($value === 'all') {
                $policies[] = $feature . '=(*)';
            }
        }
        
        return !empty($policies) ? implode(', ', $policies) : '';
    }
    
    /**
     * Get default feature policies
     */
    public function get_default_features() {
        return array(
            'geolocation' => 'none',
            'microphone' => 'none',
            'camera' => 'none',
            'payment' => 'none',
            'usb' => 'none',
            'magnetometer' => 'none',
            'gyroscope' => 'none',
            'accelerometer' => 'none',
            'ambient-light-sensor' => 'none',
        );
    }
    
    /**
     * Get available features
     */
    public function get_features() {
        return array(
            'geolocation' => 'Geolocation',
            'microphone' => 'Microphone',
            'camera' => 'Camera',
            'payment' => 'Payment',
            'usb' => 'USB',
            'magnetometer' => 'Magnetometer',
            'gyroscope' => 'Gyroscope',
            'accelerometer' => 'Accelerometer',
            'ambient-light-sensor' => 'Ambient Light Sensor',
        );
    }
    
    /**
     * Get available policy values
     */
    public function get_policy_values() {
        return array(
            'none' => 'Disabled (Recommended)',
            'self' => 'Same Origin Only',
            'all' => 'All Origins',
        );
    }
}
