<?php
/**
 * Strict-Transport-Security (HSTS) Header
 * Forces HTTPS connections
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-base-header.php';

/**
 * HSTS Header Implementation
 */
class SBSEC_HSTS extends SBSEC_Base_Header {
    
    /**
     * Constructor
     */
    public function __construct($options) {
        parent::__construct($options);
        $this->header_name = 'Strict-Transport-Security';
    }
    
    /**
     * Check if header is enabled
     */
    public function is_enabled() {
        return !empty($this->options['security_headers_hsts']) && is_ssl();
    }
    
    /**
     * Get header name
     */
    public function get_name() {
        return 'Strict-Transport-Security (HSTS)';
    }
    
    /**
     * Get header description
     */
    public function get_description() {
        return 'Forces browsers to only connect to your site over HTTPS. Only works if your site has a valid SSL certificate.';
    }
    
    /**
     * Get header value
     */
    protected function get_value() {
        // Get max-age (in seconds)
        $max_age = isset($this->options['security_headers_hsts_max_age']) 
            ? intval($this->options['security_headers_hsts_max_age']) 
            : 31536000; // Default: 1 year
        
        $value = 'max-age=' . $max_age;
        
        // Include subdomains?
        if (!empty($this->options['security_headers_hsts_subdomains'])) {
            $value .= '; includeSubDomains';
        }
        
        // Preload?
        if (!empty($this->options['security_headers_hsts_preload'])) {
            $value .= '; preload';
        }
        
        return $value;
    }
    
    /**
     * Get available durations
     */
    public function get_durations() {
        return array(
            3600 => '1 hour',
            86400 => '1 day',
            604800 => '1 week',
            2592000 => '30 days',
            7776000 => '3 months',
            15552000 => '6 months',
            31536000 => '1 year (Recommended)',
            63072000 => '2 years',
        );
    }
}
