<?php
/**
 * Base Security Header Class
 * Abstract class for all security header implementations
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Abstract Base Class for Security Headers
 */
abstract class SBSEC_Base_Header {
    
    /**
     * Plugin options
     */
    protected $options;
    
    /**
     * Header name (e.g., 'X-Frame-Options')
     */
    protected $header_name;
    
    /**
     * Constructor
     */
    public function __construct($options) {
        $this->options = $options;
    }
    
    /**
     * Check if this header is enabled
     */
    abstract public function is_enabled();
    
    /**
     * Get the header name
     */
    abstract public function get_name();
    
    /**
     * Get the header description
     */
    abstract public function get_description();
    
    /**
     * Get the header value to send
     */
    abstract protected function get_value();
    
    /**
     * Send the HTTP header
     */
    public function send() {
        if (!$this->is_enabled()) {
            return false;
        }
        
        $value = $this->get_value();
        
        if (empty($value)) {
            return false;
        }
        
        // Check if header already sent
        if ($this->header_already_sent()) {
            return false;
        }
        
        // Send the header
        header($this->header_name . ': ' . $value, true);
        
        return true;
    }
    
    /**
     * Check if header is already in headers list
     */
    protected function header_already_sent() {
        $headers = headers_list();
        
        foreach ($headers as $header) {
            if (stripos($header, $this->header_name . ':') === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Test if header is working by making a HEAD request
     */
    public function test() {
        $response = wp_remote_head(home_url(), array(
            'timeout' => 5,
            'sslverify' => false,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'status' => 'error',
                'message' => $response->get_error_message(),
            );
        }
        
        $headers = wp_remote_retrieve_headers($response);
        
        // Check if our header is present
        $header_key = strtolower(str_replace('-', '_', $this->header_name));
        
        if (isset($headers[$this->header_name]) || isset($headers[$header_key])) {
            return array(
                'status' => 'active',
                'message' => 'Header is present and working',
                'value' => $headers[$this->header_name] ?? $headers[$header_key] ?? '',
            );
        }
        
        if ($this->is_enabled()) {
            return array(
                'status' => 'warning',
                'message' => 'Header is enabled but not detected (may be overridden by server)',
            );
        }
        
        return array(
            'status' => 'inactive',
            'message' => 'Header is not enabled',
        );
    }
    
    /**
     * Sanitize header value
     */
    protected function sanitize_value($value) {
        // Remove any newlines or carriage returns to prevent header injection
        $value = str_replace(array("\r", "\n"), '', $value);
        
        return trim($value);
    }
}
