<?php
/**
 * Security Logs Class
 * Handles logging of security events
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Security_Logs {
    
    private $options;
    
    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Schedule log cleanup
        add_action('wp_scheduled_delete', array($this, 'cleanup_old_logs'));
    }
    
    /**
     * Log security event
     */
    public static function log($event_type, $severity, $ip_address = '', $user_id = 0, $message = '', $data = null) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_logs';
        
        if (empty($ip_address)) {
            $ip_address = self::get_client_ip();
        }
        
        if (empty($user_id)) {
            $user_id = get_current_user_id();
        }
        
        $wpdb->insert($table_name, array(
            'timestamp' => current_time('mysql'),
            'event_type' => $event_type,
            'severity' => $severity,
            'ip_address' => $ip_address,
            'user_id' => $user_id,
            'message' => $message,
            'data' => is_array($data) ? json_encode($data) : $data,
        ));
    }
    
    /**
     * Get logs
     */
    public static function get_logs($event_types = array(), $limit = 100, $offset = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_logs';
        
        $sql = "SELECT * FROM $table_name";
        
        if (!empty($event_types)) {
            $placeholders = implode(',', array_fill(0, count($event_types), '%s'));
            $sql .= $wpdb->prepare(" WHERE event_type IN ($placeholders)", $event_types);
        }
        
        $sql .= " ORDER BY timestamp DESC LIMIT %d OFFSET %d";
        
        return $wpdb->get_results($wpdb->prepare($sql, $limit, $offset));
    }
    
    /**
     * Get log statistics
     */
    public static function get_statistics($days = 7) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_logs';
        
        $since = date('Y-m-d H:i:s', strtotime("-$days days"));
        
        $stats = array(
            'total_events' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE timestamp > %s", $since
            )),
            'by_type' => $wpdb->get_results($wpdb->prepare(
                "SELECT event_type, COUNT(*) as count FROM $table_name WHERE timestamp > %s GROUP BY event_type", $since
            ), ARRAY_A),
            'by_severity' => $wpdb->get_results($wpdb->prepare(
                "SELECT severity, COUNT(*) as count FROM $table_name WHERE timestamp > %s GROUP BY severity", $since
            ), ARRAY_A),
            'top_ips' => $wpdb->get_results($wpdb->prepare(
                "SELECT ip_address, COUNT(*) as count FROM $table_name WHERE timestamp > %s GROUP BY ip_address ORDER BY count DESC LIMIT 10", $since
            ), ARRAY_A),
        );
        
        return $stats;
    }
    
    /**
     * Cleanup old logs
     */
    public function cleanup_old_logs() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_logs';
        
        $retention_days = $this->options['log_retention_days'] ?? 30;
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-$retention_days days"));
        
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE timestamp < %s", $cutoff_date
        ));
        
        if ($deleted > 0) {
            self::log('log_cleanup', 'low', '', 0, "Deleted $deleted old log entries");
        }
    }
    
    /**
     * Get client IP address
     */
    private static function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = $_SERVER[$key];
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Export logs to CSV
     */
    public static function export_logs($event_types = array(), $days = 30) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_logs';
        
        $since = date('Y-m-d H:i:s', strtotime("-$days days"));
        
        $sql = "SELECT * FROM $table_name WHERE timestamp > %s";
        
        if (!empty($event_types)) {
            $placeholders = implode(',', array_fill(0, count($event_types), '%s'));
            $sql .= $wpdb->prepare(" AND event_type IN ($placeholders)", $event_types);
        }
        
        $sql .= " ORDER BY timestamp DESC";
        
        $logs = $wpdb->get_results($wpdb->prepare($sql, $since), ARRAY_A);
        
        return $logs;
    }
}
