<?php
/**
 * Security Headers Manager
 * Implements HTTP security headers to protect against common vulnerabilities
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Security Headers Class
 */
class SBSEC_Security_Headers {
    
    /**
     * Plugin options
     */
    private $options;
    
    /**
     * Available header types
     */
    private $headers = array();
    
    /**
     * Constructor
     */
    public function __construct($options) {
        $this->options = $options;
        $this->init_headers();
        $this->add_hooks();
    }
    
    /**
     * Initialize all security header types
     */
    private function init_headers() {
        // Load header classes
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-xframe-options.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-xss-protection.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-content-type-options.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-hsts.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-referrer-policy.php';
        require_once SBSEC_PLUGIN_DIR . 'includes/security-headers/class-permissions-policy.php';
        
        // Initialize header instances
        $this->headers = array(
            'xframe' => new SBSEC_XFrame_Options($this->options),
            'xss' => new SBSEC_XSS_Protection($this->options),
            'content_type' => new SBSEC_Content_Type_Options($this->options),
            'hsts' => new SBSEC_HSTS($this->options),
            'referrer' => new SBSEC_Referrer_Policy($this->options),
            'permissions' => new SBSEC_Permissions_Policy($this->options),
        );
    }
    
    /**
     * Add WordPress hooks
     */
    private function add_hooks() {
        // Send headers on every request
        add_action('send_headers', array($this, 'send_security_headers'));
        
        // Admin hooks for testing
        add_action('admin_init', array($this, 'test_headers'));
    }
    
    /**
     * Send all enabled security headers
     */
    public function send_security_headers() {
        // Don't send headers if already sent
        if (headers_sent()) {
            return;
        }
        
        // Send each enabled header
        foreach ($this->headers as $header) {
            $header->send();
        }
    }
    
    /**
     * Test current headers configuration
     */
    public function test_headers() {
        // Only run on security settings page
        if (!isset($_GET['page']) || $_GET['page'] !== 'smoothbyte-security') {
            return;
        }
        
        // Store test results in transient for display
        $results = array();
        
        foreach ($this->headers as $key => $header) {
            $results[$key] = $header->test();
        }
        
        set_transient('sbsec_header_test_results', $results, 60);
    }
    
    /**
     * Get all headers status
     */
    public function get_headers_status() {
        $status = array();
        
        foreach ($this->headers as $key => $header) {
            $status[$key] = array(
                'enabled' => $header->is_enabled(),
                'name' => $header->get_name(),
                'description' => $header->get_description(),
                'status' => $header->test(),
            );
        }
        
        return $status;
    }
    
    /**
     * Get security score based on enabled headers
     */
    public function get_security_score() {
        $enabled = 0;
        $total = count($this->headers);
        
        foreach ($this->headers as $header) {
            if ($header->is_enabled()) {
                $enabled++;
            }
        }
        
        return round(($enabled / $total) * 100);
    }
}
