<?php
/**
 * Security Hardening Class
 * Implements WordPress security best practices and hardening measures
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Security_Hardening {
    
    private $options;
    
    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Disable file editing
        if (!empty($this->options['hardening_disable_file_editing'])) {
            if (!defined('DISALLOW_FILE_EDIT')) {
                define('DISALLOW_FILE_EDIT', true);
            }
        }
        
        // Disable XML-RPC
        if (!empty($this->options['hardening_disable_xmlrpc'])) {
            add_filter('xmlrpc_enabled', '__return_false');
        }
        
        // Remove WordPress version
        if (!empty($this->options['hardening_remove_wp_version'])) {
            remove_action('wp_head', 'wp_generator');
            add_filter('the_generator', '__return_empty_string');
        }
        
        // Disable directory browsing
        if (!empty($this->options['hardening_disable_directory_browsing'])) {
            add_action('init', array($this, 'disable_directory_browsing'));
        }
        
        // Disable user enumeration
        if (!empty($this->options['hardening_disable_user_enumeration'])) {
            add_action('init', array($this, 'disable_user_enumeration'));
        }
        
        // Add security headers
        add_action('send_headers', array($this, 'add_security_headers'));
        
        // Protect wp-config.php
        if (!empty($this->options['hardening_protect_config'])) {
            add_action('init', array($this, 'protect_wp_config'));
        }
        
        // Force SSL for admin
        if (!empty($this->options['hardening_force_ssl_admin']) && !defined('FORCE_SSL_ADMIN')) {
            define('FORCE_SSL_ADMIN', true);
        }
    }
    
    /**
     * Disable directory browsing
     */
    public function disable_directory_browsing() {
        $htaccess = ABSPATH . '.htaccess';
        
        if (is_writable($htaccess)) {
            $content = file_get_contents($htaccess);
            
            if (strpos($content, 'Options -Indexes') === false) {
                $rule = "\n# Disable directory browsing\nOptions -Indexes\n";
                file_put_contents($htaccess, $rule . $content);
            }
        }
    }
    
    /**
     * Disable user enumeration via REST API and author archives
     */
    public function disable_user_enumeration() {
        // Block user enumeration via REST API
        add_filter('rest_endpoints', function($endpoints) {
            if (isset($endpoints['/wp/v2/users'])) {
                unset($endpoints['/wp/v2/users']);
            }
            if (isset($endpoints['/wp/v2/users/(?P<id>[\d]+)'])) {
                unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
            }
            return $endpoints;
        });
        
        // Block author enumeration via query strings
        if (isset($_GET['author']) && is_numeric($_GET['author'])) {
            wp_die('Author enumeration is not allowed', 'Forbidden', array('response' => 403));
        }
    }
    
    /**
     * Add security headers
     */
    public function add_security_headers() {
        if (!is_admin()) {
            header('X-Frame-Options: SAMEORIGIN');
            header('X-Content-Type-Options: nosniff');
            header('X-XSS-Protection: 1; mode=block');
            header('Referrer-Policy: strict-origin-when-cross-origin');
            header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
        }
    }
    
    /**
     * Protect wp-config.php
     */
    public function protect_wp_config() {
        $htaccess = ABSPATH . '.htaccess';
        
        if (is_writable($htaccess)) {
            $content = file_get_contents($htaccess);
            
            if (strpos($content, 'wp-config.php') === false) {
                $rule = "\n# Protect wp-config.php\n<files wp-config.php>\norder allow,deny\ndeny from all\n</files>\n";
                file_put_contents($htaccess, $rule . $content);
            }
        }
    }
    
    /**
     * Get security status
     */
    public static function get_security_status() {
        $status = array(
            'file_editing_disabled' => defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT,
            'xmlrpc_disabled' => !apply_filters('xmlrpc_enabled', true),
            'ssl_admin' => is_ssl() || (defined('FORCE_SSL_ADMIN') && FORCE_SSL_ADMIN),
            'wp_version_hidden' => !has_action('wp_head', 'wp_generator'),
        );
        
        return $status;
    }
}
