<?php
/**
 * Enhanced Malware Scanner Class
 * Scans WordPress files for malicious code with deployment server integration
 * 
 * Features:
 * - Downloads malware signatures from deployment server
 * - Scans core files, plugins, themes, uploads
 * - Reports scan results to central server
 * - Quarantine system for infected files
 * - Real-time progress updates
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Malware_Scanner {
    
    private $options;
    private $signatures = array();
    private $scan_progress = array();
    private $server_url;
    private $api_key;
    
    public function __construct($options) {
        $this->options = $options;
        $this->server_url = get_option('sbsec_update_server_url', 'https://wpdeploy.smoothbyteit.dev');
        $this->api_key = get_option('sbsec_update_api_key', '');
        $this->init_hooks();
        $this->load_signatures();
    }
    
    private function init_hooks() {
        // Schedule automatic scans
        add_action('sbsec_daily_scan', array($this, 'run_scheduled_scan'));
        
        if (!wp_next_scheduled('sbsec_daily_scan')) {
            wp_schedule_event(time(), 'daily', 'sbsec_daily_scan');
        }
        
        // AJAX handlers
        add_action('wp_ajax_sbsec_run_scan', array($this, 'ajax_run_scan'));
        add_action('wp_ajax_sbsec_update_signatures', array($this, 'ajax_update_signatures'));
        add_action('wp_ajax_sbsec_get_scan_progress', array($this, 'ajax_get_scan_progress'));
        add_action('wp_ajax_sbsec_quarantine_file', array($this, 'ajax_quarantine_file'));
        add_action('wp_ajax_sbsec_restore_file', array($this, 'ajax_restore_file'));
        add_action('wp_ajax_sbsec_delete_threat', array($this, 'ajax_delete_threat'));
    }
    
    /**
     * Load malware signatures from cache or server
     */
    private function load_signatures() {
        $cached = get_transient('sbsec_malware_signatures');
        
        if ($cached !== false) {
            $this->signatures = $cached;
            return;
        }
        
        // Try to download from server
        $this->update_signatures();
        
        // Fallback to basic patterns if download fails
        if (empty($this->signatures)) {
            $this->signatures = $this->get_fallback_signatures();
        }
    }
    
    /**
     * Update signatures from deployment server
     */
    public function update_signatures() {
        if (empty($this->server_url) || empty($this->api_key)) {
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Cannot update signatures: Server URL or API key not configured');
            return false;
        }
        
        $url = rtrim($this->server_url, '/') . '/api.php/malware-signatures';
        
        $response = wp_remote_get($url, array(
            'headers' => array(
                'X-API-Key' => $this->api_key,
            ),
            'timeout' => 30,
            'sslverify' => false, // For development - remove in production
        ));
        
        if (is_wp_error($response)) {
            $error_msg = $response->get_error_message();
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Failed to update malware signatures: ' . $error_msg);
            return false;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($status_code !== 200) {
            $error_msg = $data['error'] ?? 'HTTP ' . $status_code;
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Failed to update signatures: ' . $error_msg . ' (URL: ' . $url . ')');
            return false;
        }
        
        if (!empty($data['success']) && !empty($data['signatures'])) {
            $this->signatures = $data['signatures'];
            set_transient('sbsec_malware_signatures', $this->signatures, DAY_IN_SECONDS);
            update_option('sbsec_signatures_version', $data['version'] ?? '1.0');
            update_option('sbsec_signatures_updated', current_time('mysql'));
            
            $count = $this->count_patterns($data['signatures']);
            SBSEC_Security_Logs::log('signature_update', 'info', '', 0, 'Malware signatures updated successfully (' . $count . ' patterns)');
            return true;
        }
        
        SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Invalid response from server: ' . substr($body, 0, 200));
        return false;
    }
    
    private function count_patterns($signatures) {
        $count = 0;
        foreach ($signatures as $category) {
            if (is_array($category)) {
                $count += count($category);
            }
        }
        return $count;
    }
    
    /**
     * Get fallback signatures if server unavailable
     */
    private function get_fallback_signatures() {
        return array(
            'backdoors' => array(
                array('name' => 'c99_shell', 'pattern' => 'c99|c999', 'severity' => 'critical'),
                array('name' => 'r57_shell', 'pattern' => 'r57shell|r57|r00t', 'severity' => 'critical'),
                array('name' => 'wso_shell', 'pattern' => 'wso\s*shell|wsoshell', 'severity' => 'critical'),
            ),
            'malicious_functions' => array(
                array('name' => 'eval_base64', 'pattern' => 'eval\s*\(\s*base64_decode', 'severity' => 'high'),
                array('name' => 'eval_gzinflate', 'pattern' => 'eval\s*\(\s*(gzinflate|gzuncompress)', 'severity' => 'high'),
            ),
        );
    }
    
    /**
     * Run scheduled scan
     */
    public function run_scheduled_scan() {
        $results = $this->run_scan();
        
        // Send email notification if issues found
        if (!empty($this->options['scan_email_notifications']) && !empty($results['threats'])) {
            $this->send_scan_notification($results);
        }
    }
    
    /**
     * Run comprehensive malware scan
     */
    public function run_scan($progressive = false) {
        // Initialize scan progress
        $this->scan_progress = array(
            'status' => 'running',
            'current_file' => '',
            'files_scanned' => 0,
            'total_files' => 0,
            'threats_found' => 0,
            'progress_percent' => 0,
        );
        
        update_option('sbsec_scan_progress', $this->scan_progress);
        
        $results = array(
            'scan_time' => current_time('mysql'),
            'threats' => array(),
            'warnings' => array(),
            'files_scanned' => 0,
            'scan_duration' => 0,
            'signatures_used' => count($this->get_all_patterns()),
        );
        
        $start_time = microtime(true);
        
        // Scan WordPress core files
        if (!empty($this->options['scan_wp_core'])) {
            $this->update_progress('Scanning WordPress core files...');
            $this->scan_directory(ABSPATH . 'wp-admin', $results);
            $this->scan_directory(ABSPATH . 'wp-includes', $results);
            $this->check_core_integrity($results);
        }
        
        // Scan plugins
        if (!empty($this->options['scan_plugins'])) {
            $this->update_progress('Scanning plugins...');
            $this->scan_directory(WP_PLUGIN_DIR, $results);
        }
        
        // Scan themes
        if (!empty($this->options['scan_themes'])) {
            $this->update_progress('Scanning themes...');
            $themes_dir = get_theme_root();
            $this->scan_directory($themes_dir, $results);
        }
        
        // Scan uploads directory for PHP files (shouldn't be there!)
        if (!empty($this->options['scan_uploads'])) {
            $this->update_progress('Scanning uploads directory...');
            $upload_dir = wp_upload_dir();
            if (!empty($upload_dir['basedir'])) {
                $this->scan_directory($upload_dir['basedir'], $results, array('php', 'phtml', 'php3', 'php4', 'php5', 'suspected'));
            }
        }
        
        $results['scan_duration'] = round(microtime(true) - $start_time, 2);
        
        // Save scan results locally
        update_option('sbsec_last_scan', $results);
        $this->save_scan_to_database($results);
        
        // Report to deployment server
        $this->report_scan_to_server($results);
        
        // Log scan completion
        $threat_count = count($results['threats']);
        SBSEC_Security_Logs::log('malware_scan', $threat_count > 0 ? 'high' : 'low', '', 0, "Malware scan completed. Files: {$results['files_scanned']}, Threats: $threat_count");
        
        // Update progress to complete
        $this->scan_progress['status'] = 'completed';
        $this->scan_progress['progress_percent'] = 100;
        update_option('sbsec_scan_progress', $this->scan_progress);
        
        return $results;
    }
    
    /**
     * Update scan progress
     */
    private function update_progress($message) {
        $this->scan_progress['current_file'] = $message;
        $this->scan_progress['threats_found'] = count($this->scan_progress['threats'] ?? array());
        update_option('sbsec_scan_progress', $this->scan_progress);
    }
    
    /**
     * Check WordPress core file integrity
     */
    private function check_core_integrity(&$results) {
        // This would check against WordPress.org checksums
        // For now, we'll add this as a placeholder
        // In a full implementation, this would download checksums and compare
    }
    
    /**
     * Report scan results to deployment server
     */
    private function report_scan_to_server($results) {
        if (empty($this->server_url) || empty($this->api_key)) {
            return false;
        }
        
        $response = wp_remote_post(
            $this->server_url . '/api.php/report-scan',
            array(
                'headers' => array(
                    'X-API-Key' => $this->api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'site_name' => get_bloginfo('name'),
                    'scan_results' => $results,
                )),
                'timeout' => 30,
            )
        );
        
        if (is_wp_error($response)) {
            SBSEC_Security_Logs::log('scan_report', 'error', '', 0, 'Failed to report scan: ' . $response->get_error_message());
            return false;
        }
        
        return true;
    }
    
    /**
     * Save scan results to local database
     */
    private function save_scan_to_database($results) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'sbsec_scan_results';
        
        $wpdb->insert(
            $table,
            array(
                'scan_time' => $results['scan_time'],
                'files_scanned' => $results['files_scanned'],
                'threats_found' => count($results['threats']),
                'warnings_found' => count($results['warnings']),
                'scan_duration' => $results['scan_duration'],
                'scan_data' => json_encode($results),
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%d', '%d', '%d', '%f', '%s', '%s')
        );
    }
    
    /**
     * Get all patterns from signatures
     */
    private function get_all_patterns() {
        $patterns = array();
        
        foreach ($this->signatures as $category => $sigs) {
            if (is_array($sigs)) {
                $patterns = array_merge($patterns, $sigs);
            }
        }
        
        return $patterns;
    }
    
    /**
     * Scan a directory for malware
     */
    private function scan_directory($directory, &$results, $extensions = array('php', 'phtml', 'php3', 'php4', 'php5')) {
        if (!is_dir($directory)) {
            return;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $extension = strtolower($file->getExtension());
                
                if (in_array($extension, $extensions)) {
                    $results['files_scanned']++;
                    $this->scan_file($file->getPathname(), $results);
                }
            }
        }
    }
    
    /**
     * Enhanced scan of single file with all signatures
     */
    private function scan_file($file_path, &$results) {
        $content = @file_get_contents($file_path);
        
        if ($content === false) {
            return;
        }
        
        // Update progress
        $this->scan_progress['files_scanned']++;
        $this->scan_progress['current_file'] = basename($file_path);
        
        // Periodically update progress option (every 10 files)
        if ($this->scan_progress['files_scanned'] % 10 === 0) {
            update_option('sbsec_scan_progress', $this->scan_progress);
        }
        
        // Scan against all signature categories
        foreach ($this->signatures as $category => $patterns) {
            if (!is_array($patterns)) continue;
            
            foreach ($patterns as $sig) {
                if (empty($sig['pattern'])) continue;
                
                $pattern = '/' . $sig['pattern'] . '/i';
                
                if (@preg_match($pattern, $content, $matches)) {
                    $threat = array(
                        'file' => $file_path,
                        'file_size' => filesize($file_path),
                        'file_modified' => date('Y-m-d H:i:s', filemtime($file_path)),
                        'type' => $category,
                        'signature_name' => $sig['name'],
                        'severity' => $sig['severity'],
                        'description' => $sig['description'] ?? '',
                        'match' => substr($matches[0], 0, 100),
                        'line_number' => $this->get_line_number($content, $matches[0]),
                        'detected_at' => current_time('mysql'),
                    );
                    
                    // Add to appropriate array based on severity
                    if (in_array($sig['severity'], array('critical', 'high'))) {
                        $results['threats'][] = $threat;
                    } else {
                        $results['warnings'][] = $threat;
                    }
                    
                    // For critical threats, log immediately
                    if ($sig['severity'] === 'critical') {
                        SBSEC_Security_Logs::log(
                            'threat_detected',
                            'critical',
                            '',
                            0,
                            "Critical threat detected: {$sig['name']} in " . basename($file_path)
                        );
                    }
                }
            }
        }
    }
    
    /**
     * Get line number of match in content
     */
    private function get_line_number($content, $match) {
        $pos = strpos($content, $match);
        if ($pos === false) return 0;
        
        return substr_count($content, "\n", 0, $pos) + 1;
    }
    
    /**
     * Quarantine a file
     */
    public function quarantine_file($file_path) {
        global $wpdb;
        
        if (!file_exists($file_path)) {
            return array('success' => false, 'message' => 'File not found');
        }
        
        // Create quarantine directory
        $quarantine_dir = WP_CONTENT_DIR . '/sbsec-quarantine';
        if (!file_exists($quarantine_dir)) {
            wp_mkdir_p($quarantine_dir);
            // Protect quarantine directory
            file_put_contents($quarantine_dir . '/.htaccess', 'Deny from all');
            file_put_contents($quarantine_dir . '/index.php', '<?php // Silence is golden');
        }
        
        // Generate unique quarantine filename
        $quarantine_name = md5($file_path . time()) . '.quarantine';
        $quarantine_path = $quarantine_dir . '/' . $quarantine_name;
        
        // Read file content before moving
        $content = file_get_contents($file_path);
        
        // Move file to quarantine
        if (rename($file_path, $quarantine_path)) {
            // Save quarantine record
            $table = $wpdb->prefix . 'sbsec_quarantine';
            $wpdb->insert(
                $table,
                array(
                    'original_path' => $file_path,
                    'quarantine_path' => $quarantine_path,
                    'file_size' => strlen($content),
                    'file_hash' => md5($content),
                    'quarantined_at' => current_time('mysql'),
                    'reason' => 'Manual quarantine from scan results',
                ),
                array('%s', '%s', '%d', '%s', '%s', '%s')
            );
            
            SBSEC_Security_Logs::log('file_quarantined', 'high', '', 0, "File quarantined: $file_path");
            
            return array('success' => true, 'message' => 'File quarantined successfully');
        }
        
        return array('success' => false, 'message' => 'Failed to quarantine file');
    }
    
    /**
     * Restore file from quarantine
     */
    public function restore_file($quarantine_id) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'sbsec_quarantine';
        $record = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $quarantine_id), ARRAY_A);
        
        if (!$record) {
            return array('success' => false, 'message' => 'Quarantine record not found');
        }
        
        if (!file_exists($record['quarantine_path'])) {
            return array('success' => false, 'message' => 'Quarantined file not found');
        }
        
        // Restore file
        if (rename($record['quarantine_path'], $record['original_path'])) {
            // Delete quarantine record
            $wpdb->delete($table, array('id' => $quarantine_id), array('%d'));
            
            SBSEC_Security_Logs::log('file_restored', 'medium', '', 0, "File restored: {$record['original_path']}");
            
            return array('success' => true, 'message' => 'File restored successfully');
        }
        
        return array('success' => false, 'message' => 'Failed to restore file');
    }
    
    /**
     * Delete quarantined file permanently
     */
    public function delete_quarantined_file($quarantine_id) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'sbsec_quarantine';
        $record = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $quarantine_id), ARRAY_A);
        
        if (!$record) {
            return array('success' => false, 'message' => 'Quarantine record not found');
        }
        
        // Delete physical file
        if (file_exists($record['quarantine_path'])) {
            unlink($record['quarantine_path']);
        }
        
        // Delete database record
        $wpdb->delete($table, array('id' => $quarantine_id), array('%d'));
        
        SBSEC_Security_Logs::log('file_deleted', 'high', '', 0, "Quarantined file deleted permanently: {$record['original_path']}");
        
        return array('success' => true, 'message' => 'File deleted permanently');
    }
    
    /**
     * Send scan notification email
     */
    private function send_scan_notification($results) {
        $email = $this->options['scan_notification_email'] ?? get_option('admin_email');
        $site_name = get_bloginfo('name');
        $threat_count = count($results['threats']);
        
        $subject = "[$site_name] Security Alert: $threat_count threats detected";
        
        $message = "Security scan completed on " . $site_name . "\n\n";
        $message .= "Scan Time: " . $results['scan_time'] . "\n";
        $message .= "Files Scanned: " . $results['files_scanned'] . "\n";
        $message .= "Threats Found: " . $threat_count . "\n";
        $message .= "Scan Duration: " . $results['scan_duration'] . " seconds\n\n";
        
        if (!empty($results['threats'])) {
            $message .= "THREATS DETECTED:\n\n";
            foreach ($results['threats'] as $threat) {
                $message .= "File: " . $threat['file'] . "\n";
                $message .= "Type: " . $threat['type'] . "\n";
                $message .= "Severity: " . $threat['severity'] . "\n\n";
            }
        }
        
        $message .= "\nPlease review these threats in your WordPress admin panel.";
        
        wp_mail($email, $subject, $message);
    }
    
    /**
     * AJAX handler for manual scan
     */
    public function ajax_run_scan() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Run scan in background to avoid timeout
        set_time_limit(0);
        
        $results = $this->run_scan(true);
        wp_send_json_success($results);
    }
    
    /**
     * AJAX handler to update signatures
     */
    public function ajax_update_signatures() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Reload API key in case it was just set
        $this->server_url = get_option('sbsec_update_server_url', 'https://wpdeploy.smoothbyteit.dev');
        $this->api_key = get_option('sbsec_update_api_key', '');
        
        // Check if site is registered with deployment server
        if (empty($this->server_url) || empty($this->api_key)) {
            wp_send_json_error(
                "Not connected to deployment server.\n\n" .
                "To enable signature updates:\n" .
                "1. Go to Security → Updates tab\n" .
                "2. Click 'Request Access (Free)'\n" .
                "3. Site will be registered automatically\n\n" .
                "Using fallback signatures for now."
            );
            return;
        }
        
        $success = $this->update_signatures();
        
        if ($success) {
            $version = get_option('sbsec_signatures_version', '1.0');
            $updated = get_option('sbsec_signatures_updated', 'Never');
            
            wp_send_json_success(array(
                'message' => 'Signatures updated successfully',
                'version' => $version,
                'updated' => $updated,
                'count' => count($this->get_all_patterns()),
            ));
        } else {
            wp_send_json_error('Failed to update signatures. Check connection to deployment server.');
        }
    }
    
    /**
     * AJAX handler to get scan progress
     */
    public function ajax_get_scan_progress() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $progress = get_option('sbsec_scan_progress', array());
        wp_send_json_success($progress);
    }
    
    /**
     * AJAX handler to quarantine file
     */
    public function ajax_quarantine_file() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
        
        if (empty($file_path)) {
            wp_send_json_error('File path required');
            return;
        }
        
        $result = $this->quarantine_file($file_path);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX handler to restore file
     */
    public function ajax_restore_file() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $quarantine_id = isset($_POST['quarantine_id']) ? intval($_POST['quarantine_id']) : 0;
        
        if (empty($quarantine_id)) {
            wp_send_json_error('Quarantine ID required');
            return;
        }
        
        $result = $this->restore_file($quarantine_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX handler to delete threat
     */
    public function ajax_delete_threat() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : '';
        
        if ($action_type === 'quarantine') {
            $quarantine_id = isset($_POST['quarantine_id']) ? intval($_POST['quarantine_id']) : 0;
            $result = $this->delete_quarantined_file($quarantine_id);
        } else {
            $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
            
            if (empty($file_path) || !file_exists($file_path)) {
                wp_send_json_error('Invalid file path');
                return;
            }
            
            // Delete file directly
            if (unlink($file_path)) {
                SBSEC_Security_Logs::log('file_deleted', 'high', '', 0, "Threat file deleted: $file_path");
                $result = array('success' => true, 'message' => 'File deleted successfully');
            } else {
                $result = array('success' => false, 'message' => 'Failed to delete file');
            }
        }
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * Get last scan results
     */
    public static function get_last_scan() {
        return get_option('sbsec_last_scan', array());
    }
    
    /**
     * Get quarantined files
     */
    public static function get_quarantined_files() {
        global $wpdb;
        $table = $wpdb->prefix . 'sbsec_quarantine';
        return $wpdb->get_results("SELECT * FROM $table ORDER BY quarantined_at DESC", ARRAY_A);
    }
    
    /**
     * Get scan history
     */
    public static function get_scan_history($limit = 10) {
        global $wpdb;
        $table = $wpdb->prefix . 'sbsec_scan_results';
        return $wpdb->get_results($wpdb->prepare("SELECT * FROM $table ORDER BY scan_time DESC LIMIT %d", $limit), ARRAY_A);
    }
}
