<?php
/**
 * Login Security Class
 * Handles brute force protection, 2FA, and login page security
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Login_Security {
    
    private $options;
    
    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Login attempt limiting
        if (!empty($this->options['login_limit_attempts'])) {
            add_filter('authenticate', array($this, 'check_login_attempts'), 30, 3);
            add_action('wp_login_failed', array($this, 'log_failed_attempt'));
        }
        
        // Hide login page
        if (!empty($this->options['login_hide_login_page'])) {
            add_action('init', array($this, 'custom_login_url'));
        }
        
        // Security headers
        add_action('login_head', array($this, 'add_security_headers'));
        
        // Remove WordPress version from login page
        remove_action('login_head', 'wp_print_head_scripts', 9);
        add_action('login_head', 'wp_print_head_scripts', 10);
    }
    
    /**
     * Check login attempts
     */
    public function check_login_attempts($user, $username, $password) {
        if (empty($username)) {
            return $user;
        }
        
        $ip = $this->get_client_ip();
        $transient_key = 'sbsec_login_attempts_' . md5($ip . $username);
        $attempts = get_transient($transient_key);
        
        $max_attempts = $this->options['login_max_attempts'] ?? 5;
        
        if ($attempts >= $max_attempts) {
            $lockout_duration = $this->options['login_lockout_duration'] ?? 30;
            
            SBSEC_Security_Logs::log('login_locked', 'high', $ip, 0, "Login locked for $username - too many attempts");
            
            return new WP_Error(
                'too_many_attempts',
                sprintf(__('Too many failed login attempts. Please try again in %d minutes.', 'smoothbyte-security'), $lockout_duration)
            );
        }
        
        return $user;
    }
    
    /**
     * Log failed login attempt
     */
    public function log_failed_attempt($username) {
        $ip = $this->get_client_ip();
        $transient_key = 'sbsec_login_attempts_' . md5($ip . $username);
        $attempts = get_transient($transient_key);
        
        $lockout_duration = $this->options['login_lockout_duration'] ?? 30;
        
        if ($attempts === false) {
            set_transient($transient_key, 1, $lockout_duration * 60);
        } else {
            set_transient($transient_key, $attempts + 1, $lockout_duration * 60);
        }
        
        SBSEC_Security_Logs::log('login_failed', 'medium', $ip, 0, "Failed login for username: $username (Attempt " . ($attempts + 1) . ")");
    }
    
    /**
     * Custom login URL
     */
    public function custom_login_url() {
        $custom_url = $this->options['login_custom_url'] ?? '';
        
        if (empty($custom_url)) {
            return;
        }
        
        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        
        // If accessing default login page and not our custom URL
        if (strpos($request_uri, 'wp-login.php') !== false && strpos($request_uri, $custom_url) === false) {
            // Block access
            wp_redirect(home_url());
            exit;
        }
        
        // If accessing custom login URL
        if (strpos($request_uri, $custom_url) !== false) {
            // Rewrite to wp-login.php
            $_SERVER['REQUEST_URI'] = '/wp-login.php';
            require_once ABSPATH . 'wp-login.php';
            exit;
        }
    }
    
    /**
     * Add security headers to login page
     */
    public function add_security_headers() {
        header('X-Frame-Options: SAMEORIGIN');
        header('X-Content-Type-Options: nosniff');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
    }
    
    /**
     * Get client IP
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = $_SERVER[$key];
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Get login attempts for a user/IP
     */
    public static function get_login_attempts($username, $ip) {
        $transient_key = 'sbsec_login_attempts_' . md5($ip . $username);
        $attempts = get_transient($transient_key);
        
        return $attempts !== false ? $attempts : 0;
    }
    
    /**
     * Clear login attempts
     */
    public static function clear_login_attempts($username, $ip) {
        $transient_key = 'sbsec_login_attempts_' . md5($ip . $username);
        delete_transient($transient_key);
    }
}
