<?php
/**
 * Firewall Protection Class
 * Handles IP blocking, rate limiting, and malicious request filtering
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Firewall {
    
    private $options;
    private $blocked_patterns = array(
        '/(union.*select|select.*from|insert.*into|delete.*from|drop.*table)/i',
        '/(<script|javascript:|onerror=|onload=)/i',
        '/(\.\.\/|\.\.\\\\|etc\/passwd|proc\/self)/i',
        '/(\$_GET|\$_POST|\$_REQUEST|\$_SERVER|\$GLOBALS)/i'
    );
    
    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Early firewall check - runs before WordPress fully loads
        add_action('init', array($this, 'firewall_check'), 1);
        add_action('wp_login_failed', array($this, 'log_failed_login'));
        
        // Block bad user agents
        add_action('init', array($this, 'block_bad_user_agents'));
    }
    
    /**
     * Main firewall check
     */
    public function firewall_check() {
        $ip = $this->get_client_ip();
        
        // Check if IP is blocked
        if ($this->is_ip_blocked($ip)) {
            $this->block_request('IP address is blocked', $ip);
        }
        
        // Rate limiting
        if (!empty($this->options['firewall_rate_limiting'])) {
            $this->check_rate_limit($ip);
        }
        
        // Check for malicious requests
        if (!empty($this->options['firewall_block_bad_requests'])) {
            $this->check_malicious_requests();
        }
    }
    
    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = $_SERVER[$key];
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Check if IP is blocked
     */
    private function is_ip_blocked($ip) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_blocked_ips';
        
        $blocked = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE ip_address = %s AND (expires_at IS NULL OR expires_at > NOW())",
            $ip
        ));
        
        return !empty($blocked);
    }
    
    /**
     * Block IP address
     */
    public function block_ip($ip, $reason, $duration = null, $type = 'auto') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_blocked_ips';
        
        $expires_at = $duration ? date('Y-m-d H:i:s', time() + ($duration * 60)) : null;
        
        $wpdb->replace($table_name, array(
            'ip_address' => $ip,
            'reason' => $reason,
            'blocked_at' => current_time('mysql'),
            'expires_at' => $expires_at,
            'block_type' => $type
        ));
        
        // Log the event
        SBSEC_Security_Logs::log('firewall_block', 'high', $ip, 0, "IP blocked: $reason");
    }
    
    /**
     * Rate limiting check
     */
    private function check_rate_limit($ip) {
        $transient_key = 'sbsec_rate_' . md5($ip);
        $requests = get_transient($transient_key);
        
        if ($requests === false) {
            set_transient($transient_key, 1, 60); // 1 minute
        } else {
            $limit = $this->options['firewall_rate_limit'] ?? 60;
            
            if ($requests >= $limit) {
                $this->block_ip($ip, 'Rate limit exceeded', 30, 'auto');
                $this->block_request('Too many requests', $ip);
            }
            
            set_transient($transient_key, $requests + 1, 60);
        }
    }
    
    /**
     * Check for malicious request patterns
     */
    private function check_malicious_requests() {
        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        $query_string = $_SERVER['QUERY_STRING'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        $check_data = $request_uri . ' ' . $query_string . ' ' . $user_agent;
        
        foreach ($this->blocked_patterns as $pattern) {
            if (preg_match($pattern, $check_data)) {
                $ip = $this->get_client_ip();
                $this->block_ip($ip, 'Malicious request pattern detected', 60, 'auto');
                $this->block_request('Malicious request detected', $ip);
            }
        }
    }
    
    /**
     * Block bad user agents
     */
    public function block_bad_user_agents() {
        $bad_bots = array('semrush', 'ahrefs', 'mj12bot', 'dotbot', 'megaindex', 'blexbot');
        $user_agent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        foreach ($bad_bots as $bot) {
            if (strpos($user_agent, $bot) !== false) {
                $ip = $this->get_client_ip();
                SBSEC_Security_Logs::log('firewall_bot_block', 'medium', $ip, 0, "Bad bot blocked: $bot");
                wp_die('Access denied', 'Forbidden', array('response' => 403));
            }
        }
    }
    
    /**
     * Block request and send response
     */
    private function block_request($message, $ip) {
        SBSEC_Security_Logs::log('firewall_request_blocked', 'high', $ip, 0, $message);
        
        header('HTTP/1.1 403 Forbidden');
        die('Access Denied');
    }
    
    /**
     * Log failed login attempts
     */
    public function log_failed_login($username) {
        $ip = $this->get_client_ip();
        SBSEC_Security_Logs::log('login_failed', 'medium', $ip, 0, "Failed login attempt for username: $username");
    }
    
    /**
     * Get blocked IPs list
     */
    public static function get_blocked_ips() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_blocked_ips';
        
        return $wpdb->get_results("SELECT * FROM $table_name ORDER BY blocked_at DESC");
    }
    
    /**
     * Unblock IP address
     */
    public static function unblock_ip($ip) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_blocked_ips';
        
        $wpdb->delete($table_name, array('ip_address' => $ip));
        SBSEC_Security_Logs::log('firewall_unblock', 'low', $ip, 0, "IP unblocked manually");
    }
}
