<?php
/**
 * File Monitor Class
 * Monitors WordPress files for changes and unauthorized modifications
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_File_Monitor {
    
    private $options;
    
    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Schedule file integrity checks
        add_action('sbsec_hourly_check', array($this, 'check_file_integrity'));
        
        if (!wp_next_scheduled('sbsec_hourly_check')) {
            wp_schedule_event(time(), 'hourly', 'sbsec_hourly_check');
        }
        
        // AJAX handlers
        add_action('wp_ajax_sbsec_baseline_scan', array($this, 'ajax_create_baseline'));
    }
    
    /**
     * Create baseline checksums
     */
    public function create_baseline() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_file_checksums';
        
        $directories = array();
        
        if (!empty($this->options['monitor_core_files'])) {
            $directories[] = ABSPATH . 'wp-admin';
            $directories[] = ABSPATH . 'wp-includes';
        }
        
        if (!empty($this->options['monitor_plugin_files'])) {
            $directories[] = WP_PLUGIN_DIR;
        }
        
        if (!empty($this->options['monitor_theme_files'])) {
            $directories[] = get_theme_root();
        }
        
        foreach ($directories as $directory) {
            $this->scan_directory_for_baseline($directory, $table_name);
        }
        
        update_option('sbsec_baseline_created', current_time('mysql'));
        
        return true;
    }
    
    /**
     * Scan directory and create checksums
     */
    private function scan_directory_for_baseline($directory, $table_name) {
        global $wpdb;
        
        if (!is_dir($directory)) {
            return;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $file_path = $file->getPathname();
                $relative_path = str_replace(ABSPATH, '', $file_path);
                
                $checksum = md5_file($file_path);
                $file_size = $file->getSize();
                $last_modified = date('Y-m-d H:i:s', $file->getMTime());
                
                $wpdb->replace($table_name, array(
                    'file_path' => $relative_path,
                    'checksum' => $checksum,
                    'file_size' => $file_size,
                    'last_modified' => $last_modified,
                    'last_checked' => current_time('mysql'),
                ));
            }
        }
    }
    
    /**
     * Check file integrity
     */
    public function check_file_integrity() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sbsec_file_checksums';
        
        // Get all monitored files
        $files = $wpdb->get_results("SELECT * FROM $table_name");
        
        $changes = array();
        
        foreach ($files as $file) {
            $full_path = ABSPATH . $file->file_path;
            
            if (!file_exists($full_path)) {
                $changes[] = array(
                    'file' => $file->file_path,
                    'type' => 'deleted',
                    'severity' => 'high',
                );
                continue;
            }
            
            $current_checksum = md5_file($full_path);
            
            if ($current_checksum !== $file->checksum) {
                $changes[] = array(
                    'file' => $file->file_path,
                    'type' => 'modified',
                    'severity' => 'medium',
                );
                
                // Update checksum
                $wpdb->update($table_name, 
                    array(
                        'checksum' => $current_checksum,
                        'last_checked' => current_time('mysql'),
                    ),
                    array('id' => $file->id)
                );
            }
        }
        
        // Log changes
        if (!empty($changes) && !empty($this->options['monitor_alert_on_change'])) {
            foreach ($changes as $change) {
                SBSEC_Security_Logs::log('file_change', $change['severity'], '', 0, 
                    "File {$change['type']}: {$change['file']}");
            }
            
            // Send email notification
            $this->send_file_change_notification($changes);
        }
        
        return $changes;
    }
    
    /**
     * Send file change notification
     */
    private function send_file_change_notification($changes) {
        $email = $this->options['notification_email'] ?? get_option('admin_email');
        $site_name = get_bloginfo('name');
        $change_count = count($changes);
        
        $subject = "[$site_name] Security Alert: $change_count file changes detected";
        
        $message = "File integrity check on " . $site_name . "\n\n";
        $message .= "Changes detected: $change_count\n\n";
        
        foreach ($changes as $change) {
            $message .= "File: " . $change['file'] . "\n";
            $message .= "Change: " . $change['type'] . "\n";
            $message .= "Severity: " . $change['severity'] . "\n\n";
        }
        
        $message .= "\nPlease review these changes in your WordPress admin panel.";
        
        wp_mail($email, $subject, $message);
    }
    
    /**
     * AJAX handler to create baseline
     */
    public function ajax_create_baseline() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $this->create_baseline();
        wp_send_json_success(array('message' => 'Baseline created successfully'));
    }
    
    /**
     * Get file changes
     */
    public static function get_recent_changes($limit = 50) {
        return SBSEC_Security_Logs::get_logs(array('file_change'), $limit);
    }
}
