<?php
/**
 * Auto Updater Class
 * Handles automatic updates from SmoothByte deployment server
 */

if (!defined('ABSPATH')) {
    exit;
}

class SBSEC_Auto_Updater {
    
    private static $instance = null;
    private $plugin_slug = 'smoothbyte-security-pro';
    private $plugin_file;
    private $server_url;
    private $api_key;
    private $current_version;
    
    public static function get_instance() {
        return self::$instance;
    }
    
    public function __construct($plugin_file, $current_version) {
        self::$instance = $this;
        $this->plugin_file = $plugin_file;
        $this->current_version = $current_version;
        
        // Get settings from options
        $this->server_url = get_option('sbsec_update_server_url', 'https://wpdeploy.smoothbyteit.dev');
        $this->api_key = get_option('sbsec_update_api_key', '');
        
        // Only register hooks if configured
        if ($this->is_configured()) {
            $this->init_hooks();
        }
        
        // Always add settings hooks for admin
        if (is_admin()) {
            add_action('admin_init', array($this, 'register_settings'));
            add_action('admin_notices', array($this, 'admin_notices'));
        }
    }
    
    /**
     * Check if updater is configured
     */
    private function is_configured() {
        return !empty($this->server_url) && !empty($this->api_key);
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Hook into WordPress update checks
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_updates'));
        add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
        
        // Add authorization header for downloads
        add_filter('http_request_args', array($this, 'add_download_auth'), 10, 2);
        
        // Check for pending updates hourly
        add_action('sbsec_check_updates', array($this, 'check_pending_updates'));
        if (!wp_next_scheduled('sbsec_check_updates')) {
            wp_schedule_event(time(), 'hourly', 'sbsec_check_updates');
        }
        
        // Register REST API endpoint for remote trigger
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // Report health status
        add_action('sbsec_report_health', array($this, 'report_health'));
        if (!wp_next_scheduled('sbsec_report_health')) {
            wp_schedule_event(time(), 'twicedaily', 'sbsec_report_health');
        }
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('sbsec_updater', 'sbsec_update_server_url');
        register_setting('sbsec_updater', 'sbsec_update_api_key');
        register_setting('sbsec_updater', 'sbsec_auto_update', array('default' => true));
    }
    
    /**
     * Admin notices
     */
    public function admin_notices() {
        if (!$this->is_configured()) {
            $settings_url = admin_url('admin.php?page=smoothbyte-security&tab=updates');
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>SmoothByte Security Pro:</strong> Configure the update server to receive automatic updates. ';
            echo '<a href="' . esc_url($settings_url) . '">Configure Now</a></p>';
            echo '</div>';
        }
    }
    
    /**
     * Make API request to deployment server
     */
    private function api_request($endpoint, $data = array(), $method = 'POST') {
        $url = rtrim($this->server_url, '/') . '/api.php/' . ltrim($endpoint, '/');
        
        $args = array(
            'method' => $method,
            'timeout' => 30,
            'headers' => array(
                'X-API-Key' => $this->api_key,
                'Content-Type' => 'application/json',
            ),
        );
        
        if ($method === 'POST' && !empty($data)) {
            $args['body'] = json_encode($data);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }
    
    /**
     * Check for updates
     */
    public function check_for_updates($transient) {
        if (empty($transient->checked) || !$this->is_configured()) {
            return $transient;
        }
        
        // Prepare plugin data
        $plugins = array(
            $this->plugin_slug => array(
                'version' => $this->current_version
            )
        );
        
        $result = $this->api_request('check-updates', array('plugins' => $plugins));
        
        if (!empty($result['updates'][$this->plugin_slug])) {
            $update = $result['updates'][$this->plugin_slug];
            
            $transient->response[$this->plugin_file] = (object) array(
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_file,
                'new_version' => $update['version'],
                'url' => !empty($update['homepage']) ? $update['homepage'] : '',
                'package' => $update['download_url'],
                'tested' => $update['tested_up_to'],
                'requires_php' => $update['requires_php'],
            );
        }
        
        return $transient;
    }
    
    /**
     * Check for pending updates from deployment server
     */
    public function check_pending_updates() {
        if (!$this->is_configured()) {
            return;
        }
        
        $result = $this->api_request('pending-updates', array(), 'GET');
        
        if (!empty($result['updates'])) {
            $auto_update = get_option('sbsec_auto_update', true);
            
            foreach ($result['updates'] as $update) {
                if ($update['plugin_slug'] === $this->plugin_slug) {
                    if ($auto_update) {
                        $this->install_update($update);
                    } else {
                        // Trigger WordPress to show update notification
                        delete_site_transient('update_plugins');
                    }
                }
            }
        }
    }
    
    /**
     * Install update
     */
    private function install_update($update) {
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        
        $skin = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        
        $result = $upgrader->upgrade($this->plugin_file);
        
        $status = $result ? 'completed' : 'failed';
        $error = $result ? '' : 'Installation failed';
        
        // Report back to server
        $this->api_request('update-deployed/' . $update['queue_id'], array(
            'status' => $status,
            'error_message' => $error,
        ));
    }
    
    /**
     * Plugin info for update screen
     */
    public function plugin_info($false, $action, $args) {
        if ($action !== 'plugin_information') {
            return $false;
        }
        
        if (!isset($args->slug) || $args->slug !== $this->plugin_slug) {
            return $false;
        }
        
        if (!$this->is_configured()) {
            return $false;
        }
        
        // Get plugin info from server
        $plugins = array($this->plugin_slug => array('version' => $this->current_version));
        $result = $this->api_request('check-updates', array('plugins' => $plugins));
        
        if (!empty($result['updates'][$this->plugin_slug])) {
            $update = $result['updates'][$this->plugin_slug];
            
            return (object) array(
                'name' => $update['name'],
                'slug' => $this->plugin_slug,
                'version' => $update['version'],
                'author' => '<a href="https://smoothbyteit.dev">SmoothByte IT</a>',
                'homepage' => 'https://smoothbyteit.dev',
                'requires' => $update['requires_wp'],
                'tested' => $update['tested_up_to'],
                'requires_php' => $update['requires_php'],
                'download_link' => $update['download_url'],
                'sections' => array(
                    'description' => 'Comprehensive WordPress security solution with malware scanning, firewall protection, and advanced hardening features.',
                ),
            );
        }
        
        return $false;
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('smoothbyte-security/v1', '/trigger-install', array(
            'methods' => 'POST',
            'callback' => array($this, 'rest_trigger_install'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));
    }
    
    /**
     * Verify API key for REST requests
     */
    public function verify_api_key($request) {
        $apiKey = $request->get_header('X-API-Key');
        return $apiKey === $this->api_key;
    }
    
    /**
     * REST API callback to trigger immediate installation
     */
    public function rest_trigger_install($request) {
        $params = $request->get_json_params();
        $pluginSlug = $params['plugin_slug'] ?? '';
        $version = $params['version'] ?? '';
        
        if (!$this->is_configured()) {
            return new WP_Error('not_configured', 'Update server not configured', array('status' => 400));
        }
        
        // Force check for updates
        delete_site_transient('update_plugins');
        $this->check_pending_updates();
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => "Plugin updated to version $version",
            'plugin' => $pluginSlug,
            'version' => $version
        ));
    }
    
    /**
     * Report site health status to deployment server
     */
    public function report_health() {
        if (!$this->is_configured()) {
            return;
        }

        $health_data = array(
            'wp_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'plugin_slug' => 'smoothbyte-security-pro',
            'plugin_version' => SBSEC_VERSION,
            'plugin_file' => SBSEC_PLUGIN_BASENAME,
            'health_status' => 'healthy',
            'plugins' => array(),
            'errors' => array()
        );

        // Get all active plugins
        $active_plugins = get_option('active_plugins', array());
        foreach ($active_plugins as $plugin) {
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
            $health_data['plugins'][] = array(
                'slug' => dirname($plugin),
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'active' => true
            );
        }

        // Report to server
        $this->api_request('report-health', $health_data);
    }
    
    /**
     * Add authentication headers to download requests
     */
    public function add_download_auth($args, $url) {
        // Only add auth for requests to our deployment server
        if (strpos($url, $this->server_url) !== false) {
            if (!isset($args['headers'])) {
                $args['headers'] = array();
            }
            $args['headers']['X-API-Key'] = $this->api_key;
        }
        return $args;
    }
    
    /**
     * Request automatic access to deployment server
     */
    public function request_access() {
        if (empty($this->server_url)) {
            return array('success' => false, 'message' => 'Server URL required');
        }
        
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        
        // Make automatic registration request
        $url = rtrim($this->server_url, '/') . '/api.php/request-access';
        
        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'site_url' => $site_url,
                'site_name' => $site_name,
            )),
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (!empty($result['success']) && !empty($result['api_key'])) {
            // Save the site-specific API key
            update_option('sbsec_update_api_key', $result['api_key']);
            $this->api_key = $result['api_key'];
            
            // Initialize hooks now that we're configured
            if ($this->is_configured()) {
                $this->init_hooks();
            }
            
            return array(
                'success' => true,
                'message' => 'Access granted! Your site is now registered with the deployment server.',
                'api_key' => $result['api_key'],
            );
        } else {
            return array(
                'success' => false,
                'message' => $result['error'] ?? 'Access request failed',
            );
        }
    }
}
