<?php
/**
 * Security Headers Admin Tab
 *
 * @package SmoothByte_Security_Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

$options = get_option('sbsec_options');
$is_ssl = is_ssl();
?>

<div class="wrap">
    <h2><?php _e('Security Headers', 'smoothbyte-security'); ?></h2>
    
    <div class="notice notice-info">
        <p>
            <strong><?php _e('What are Security Headers?', 'smoothbyte-security'); ?></strong><br>
            <?php _e('Security headers are HTTP response headers that tell browsers how to behave when handling your site\'s content. They provide an additional layer of security against common attacks.', 'smoothbyte-security'); ?>
        </p>
    </div>
    
    <?php if (!$is_ssl): ?>
    <div class="notice notice-warning">
        <p>
            <strong><?php _e('SSL/HTTPS Not Detected', 'smoothbyte-security'); ?></strong><br>
            <?php _e('Some security headers (like HSTS) require your site to be running on HTTPS. Please install an SSL certificate for maximum security.', 'smoothbyte-security'); ?>
        </p>
    </div>
    <?php endif; ?>
    
    <form method="post" action="options.php">
        <?php settings_fields('sbsec_options'); ?>
        
        <table class="form-table">
            <!-- Master Toggle -->
            <tr>
                <th scope="row">
                    <label for="security_headers"><?php _e('Enable Security Headers', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers]" 
                               id="security_headers"
                               value="1" 
                               <?php checked($options['security_headers'], true); ?> />
                        <?php _e('Enable all security headers', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Master switch to enable/disable all security headers at once.', 'smoothbyte-security'); ?>
                    </p>
                </td>
            </tr>
        </table>
        
        <hr>
        
        <h3><?php _e('Individual Headers', 'smoothbyte-security'); ?></h3>
        
        <table class="form-table">
            <!-- X-Frame-Options -->
            <tr>
                <th scope="row">
                    <label for="security_headers_xframe"><?php _e('X-Frame-Options', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_xframe]" 
                               id="security_headers_xframe"
                               value="1" 
                               <?php checked($options['security_headers_xframe'], true); ?> />
                        <?php _e('Enable X-Frame-Options header', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Prevents clickjacking attacks by controlling whether your site can be embedded in iframes.', 'smoothbyte-security'); ?>
                    </p>
                    
                    <select name="sbsec_options[security_headers_xframe_mode]" style="margin-top: 10px;">
                        <option value="DENY" <?php selected($options['security_headers_xframe_mode'], 'DENY'); ?>>
                            DENY - <?php _e('Never allow framing', 'smoothbyte-security'); ?>
                        </option>
                        <option value="SAMEORIGIN" <?php selected($options['security_headers_xframe_mode'], 'SAMEORIGIN'); ?>>
                            SAMEORIGIN - <?php _e('Only allow same origin', 'smoothbyte-security'); ?>
                        </option>
                    </select>
                </td>
            </tr>
            
            <!-- X-XSS-Protection -->
            <tr>
                <th scope="row">
                    <label for="security_headers_xss"><?php _e('X-XSS-Protection', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_xss]" 
                               id="security_headers_xss"
                               value="1" 
                               <?php checked($options['security_headers_xss'], true); ?> />
                        <?php _e('Enable X-XSS-Protection header', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Enables browser\'s built-in XSS filter (legacy browsers). Modern browsers use CSP instead.', 'smoothbyte-security'); ?>
                    </p>
                    
                    <select name="sbsec_options[security_headers_xss_mode]" style="margin-top: 10px;">
                        <option value="block" <?php selected($options['security_headers_xss_mode'], 'block'); ?>>
                            Block - <?php _e('Block page if attack detected', 'smoothbyte-security'); ?>
                        </option>
                        <option value="sanitize" <?php selected($options['security_headers_xss_mode'], 'sanitize'); ?>>
                            Sanitize - <?php _e('Remove unsafe parts', 'smoothbyte-security'); ?>
                        </option>
                    </select>
                </td>
            </tr>
            
            <!-- X-Content-Type-Options -->
            <tr>
                <th scope="row">
                    <label for="security_headers_content_type"><?php _e('X-Content-Type-Options', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_content_type]" 
                               id="security_headers_content_type"
                               value="1" 
                               <?php checked($options['security_headers_content_type'], true); ?> />
                        <?php _e('Enable X-Content-Type-Options header', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Prevents MIME-sniffing attacks. Always set to "nosniff".', 'smoothbyte-security'); ?>
                    </p>
                </td>
            </tr>
            
            <!-- HSTS -->
            <tr>
                <th scope="row">
                    <label for="security_headers_hsts"><?php _e('Strict-Transport-Security (HSTS)', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_hsts]" 
                               id="security_headers_hsts"
                               value="1" 
                               <?php checked($options['security_headers_hsts'], true); ?>
                               <?php disabled(!$is_ssl); ?> />
                        <?php _e('Enable HSTS header', 'smoothbyte-security'); ?>
                        <?php if (!$is_ssl): ?>
                            <span style="color: #dc3232;"><?php _e('(Requires SSL)', 'smoothbyte-security'); ?></span>
                        <?php endif; ?>
                    </label>
                    <p class="description">
                        <?php _e('Forces browsers to only use HTTPS. Only enable if you have a valid SSL certificate.', 'smoothbyte-security'); ?>
                    </p>
                    
                    <?php if ($is_ssl): ?>
                    <div style="margin-top: 10px;">
                        <label>
                            <input type="checkbox" 
                                   name="sbsec_options[security_headers_hsts_subdomains]" 
                                   value="1" 
                                   <?php checked($options['security_headers_hsts_subdomains'], true); ?> />
                            <?php _e('Include subdomains', 'smoothbyte-security'); ?>
                        </label>
                        <br>
                        <label style="margin-top: 5px; display: inline-block;">
                            <select name="sbsec_options[security_headers_hsts_max_age]">
                                <option value="3600" <?php selected($options['security_headers_hsts_max_age'], 3600); ?>>1 hour</option>
                                <option value="86400" <?php selected($options['security_headers_hsts_max_age'], 86400); ?>>1 day</option>
                                <option value="604800" <?php selected($options['security_headers_hsts_max_age'], 604800); ?>>1 week</option>
                                <option value="2592000" <?php selected($options['security_headers_hsts_max_age'], 2592000); ?>>30 days</option>
                                <option value="31536000" <?php selected($options['security_headers_hsts_max_age'], 31536000); ?>>1 year (Recommended)</option>
                            </select>
                            <?php _e('Cache duration', 'smoothbyte-security'); ?>
                        </label>
                    </div>
                    <?php endif; ?>
                </td>
            </tr>
            
            <!-- Referrer-Policy -->
            <tr>
                <th scope="row">
                    <label for="security_headers_referrer"><?php _e('Referrer-Policy', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_referrer]" 
                               id="security_headers_referrer"
                               value="1" 
                               <?php checked($options['security_headers_referrer'], true); ?> />
                        <?php _e('Enable Referrer-Policy header', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Controls how much referrer information is sent with requests.', 'smoothbyte-security'); ?>
                    </p>
                    
                    <select name="sbsec_options[security_headers_referrer_policy]" style="margin-top: 10px; width: 100%; max-width: 400px;">
                        <option value="no-referrer" <?php selected($options['security_headers_referrer_policy'], 'no-referrer'); ?>>
                            no-referrer
                        </option>
                        <option value="strict-origin-when-cross-origin" <?php selected($options['security_headers_referrer_policy'], 'strict-origin-when-cross-origin'); ?>>
                            strict-origin-when-cross-origin (Recommended)
                        </option>
                        <option value="same-origin" <?php selected($options['security_headers_referrer_policy'], 'same-origin'); ?>>
                            same-origin
                        </option>
                        <option value="origin" <?php selected($options['security_headers_referrer_policy'], 'origin'); ?>>
                            origin
                        </option>
                    </select>
                </td>
            </tr>
            
            <!-- Permissions-Policy -->
            <tr>
                <th scope="row">
                    <label for="security_headers_permissions"><?php _e('Permissions-Policy', 'smoothbyte-security'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" 
                               name="sbsec_options[security_headers_permissions]" 
                               id="security_headers_permissions"
                               value="1" 
                               <?php checked($options['security_headers_permissions'], true); ?> />
                        <?php _e('Enable Permissions-Policy header', 'smoothbyte-security'); ?>
                    </label>
                    <p class="description">
                        <?php _e('Controls which browser features (camera, microphone, geolocation, etc.) can be used.', 'smoothbyte-security'); ?>
                    </p>
                    <p class="description">
                        <em><?php _e('By default, all potentially sensitive features are disabled for enhanced security.', 'smoothbyte-security'); ?></em>
                    </p>
                </td>
            </tr>
        </table>
        
        <?php submit_button(__('Save Security Headers Settings', 'smoothbyte-security')); ?>
    </form>
    
    <hr>
    
    <h3><?php _e('Header Status Test', 'smoothbyte-security'); ?></h3>
    <p><?php _e('Click "Test Headers" to check which security headers are currently active on your site.', 'smoothbyte-security'); ?></p>
    
    <button type="button" class="button button-secondary" id="test-headers-btn">
        <?php _e('Test Headers', 'smoothbyte-security'); ?>
    </button>
    
    <div id="header-test-results" style="margin-top: 20px;"></div>
</div>

<script>
jQuery(document).ready(function($) {
    $('#test-headers-btn').on('click', function() {
        var $btn = $(this);
        var $results = $('#header-test-results');
        
        $btn.prop('disabled', true).text('<?php _e('Testing...', 'smoothbyte-security'); ?>');
        $results.html('<p><?php _e('Testing headers...', 'smoothbyte-security'); ?></p>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_test_headers',
                nonce: '<?php echo wp_create_nonce('sbsec_test_headers'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    var html = '<div class="notice notice-success"><p><strong><?php _e('Test Complete!', 'smoothbyte-security'); ?></strong></p></div>';
                    html += '<table class="widefat">';
                    html += '<thead><tr><th><?php _e('Header', 'smoothbyte-security'); ?></th><th><?php _e('Status', 'smoothbyte-security'); ?></th><th><?php _e('Value', 'smoothbyte-security'); ?></th></tr></thead>';
                    html += '<tbody>';
                    
                    $.each(response.data.headers, function(key, header) {
                        var statusClass = header.status === 'active' ? 'notice-success' : 'notice-warning';
                        html += '<tr>';
                        html += '<td><strong>' + header.name + '</strong></td>';
                        html += '<td><span class="' + statusClass + '" style="padding: 3px 8px; display: inline-block;">' + header.status + '</span></td>';
                        html += '<td><code>' + (header.value || 'N/A') + '</code></td>';
                        html += '</tr>';
                    });
                    
                    html += '</tbody></table>';
                    $results.html(html);
                } else {
                    $results.html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                }
            },
            error: function() {
                $results.html('<div class="notice notice-error"><p><?php _e('Error testing headers', 'smoothbyte-security'); ?></p></div>');
            },
            complete: function() {
                $btn.prop('disabled', false).text('<?php _e('Test Headers', 'smoothbyte-security'); ?>');
            }
        });
    });
});
</script>

<style>
    .notice-success { background: #d4edda; color: #155724; border-left: 4px solid #28a745; }
    .notice-warning { background: #fff3cd; color: #856404; border-left: 4px solid #ffc107; }
    .widefat th { background: #f1f1f1; }
    .widefat td { vertical-align: middle; }
</style>
