<?php
/**
 * Comprehensive Malware Scanner Admin Tab
 */

if (!defined('ABSPATH')) exit;

$last_scan = SBSEC_Malware_Scanner::get_last_scan();
$quarantined = SBSEC_Malware_Scanner::get_quarantined_files();
$scan_history = SBSEC_Malware_Scanner::get_scan_history(5);
$sig_version = get_option('sbsec_signatures_version', 'Not updated');
$sig_updated = get_option('sbsec_signatures_updated', 'Never');
?>

<div class="wrap sbsec-scanner-wrap">
    <h2><?php _e('Malware Scanner', 'smoothbyte-security'); ?></h2>
    
    <!-- Scanner Status Cards -->
    <div class="sbsec-status-cards">
        <div class="sbsec-card">
            <h3><?php _e('Last Scan', 'smoothbyte-security'); ?></h3>
            <div class="stat-value"><?php 
                if (!empty($last_scan['scan_time'])) {
                    $scan_time = strtotime($last_scan['scan_time']);
                    echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $scan_time);
                } else {
                    echo 'Never';
                }
            ?></div>
        </div>
        <div class="sbsec-card">
            <h3><?php _e('Threats Found', 'smoothbyte-security'); ?></h3>
            <div class="stat-value <?php echo !empty($last_scan['threats']) ? 'danger' : 'success'; ?>">
                <?php echo isset($last_scan['threats']) ? count($last_scan['threats']) : 0; ?>
            </div>
        </div>
        <div class="sbsec-card">
            <h3><?php _e('Files Scanned', 'smoothbyte-security'); ?></h3>
            <div class="stat-value"><?php echo $last_scan['files_scanned'] ?? 0; ?></div>
        </div>
        <div class="sbsec-card">
            <h3><?php _e('Quarantined Files', 'smoothbyte-security'); ?></h3>
            <div class="stat-value"><?php echo count($quarantined); ?></div>
        </div>
    </div>
    
    <!-- Scanner Controls -->
    <div class="sbsec-scanner-controls">
        <button type="button" class="button button-primary button-hero" id="sbsec-run-scan">
            <span class="dashicons dashicons-search"></span>
            <?php _e('Run Full Scan', 'smoothbyte-security'); ?>
        </button>
        
        <button type="button" class="button button-secondary" id="sbsec-update-signatures">
            <span class="dashicons dashicons-update"></span>
            <?php _e('Update Signatures', 'smoothbyte-security'); ?>
        </button>
        
        <div class="signature-info">
            <small>
                <?php _e('Signatures:', 'smoothbyte-security'); ?> <strong><?php echo esc_html($sig_version); ?></strong>
                | <?php _e('Updated:', 'smoothbyte-security'); ?> <?php 
                    if ($sig_updated && $sig_updated !== 'Never') {
                        $sig_time = strtotime($sig_updated);
                        echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $sig_time);
                    } else {
                        echo esc_html($sig_updated);
                    }
                ?>
            </small>
        </div>
    </div>
    
    <!-- Scan Progress -->
    <div class="sbsec-scan-progress" id="scan-progress" style="display: none;">
        <div class="progress-header">
            <h3><?php _e('Scan in Progress...', 'smoothbyte-security'); ?></h3>
            <button type="button" class="button button-small" id="cancel-scan" style="display:none;"><?php _e('Cancel', 'smoothbyte-security'); ?></button>
        </div>
        <div class="progress-bar-container">
            <div class="progress-bar" id="progress-bar"></div>
        </div>
        <div class="progress-info">
            <div id="progress-status"><?php _e('Initializing...', 'smoothbyte-security'); ?></div>
            <div id="progress-details"></div>
        </div>
    </div>
    
    <!-- Scan Results Tabs -->
    <div class="sbsec-tabs">
        <h2 class="nav-tab-wrapper">
            <a href="#threats" class="nav-tab nav-tab-active"><?php _e('Threats', 'smoothbyte-security'); ?> <?php if (!empty($last_scan['threats'])): ?><span class="count">(<?php echo count($last_scan['threats']); ?>)</span><?php endif; ?></a>
            <a href="#warnings" class="nav-tab"><?php _e('Warnings', 'smoothbyte-security'); ?> <?php if (!empty($last_scan['warnings'])): ?><span class="count">(<?php echo count($last_scan['warnings']); ?>)</span><?php endif; ?></a>
            <a href="#quarantine" class="nav-tab"><?php _e('Quarantine', 'smoothbyte-security'); ?> <?php if ($quarantined): ?><span class="count">(<?php echo count($quarantined); ?>)</span><?php endif; ?></a>
            <a href="#history" class="nav-tab"><?php _e('Scan History', 'smoothbyte-security'); ?></a>
        </h2>
        
        <!-- Threats Tab -->
        <div id="threats" class="tab-content active">
            <?php if (empty($last_scan['threats'])): ?>
                <div class="notice notice-success">
                    <p><strong><?php _e('No threats detected!', 'smoothbyte-security'); ?></strong></p>
                    <p><?php _e('Your site appears to be clean.', 'smoothbyte-security'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Severity', 'smoothbyte-security'); ?></th>
                            <th><?php _e('File', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Threat Type', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Description', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Actions', 'smoothbyte-security'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($last_scan['threats'] as $threat): ?>
                        <tr>
                            <td>
                                <span class="severity-badge severity-<?php echo esc_attr($threat['severity']); ?>">
                                    <?php echo esc_html(strtoupper($threat['severity'])); ?>
                                </span>
                            </td>
                            <td>
                                <strong><?php echo esc_html(basename($threat['file'])); ?></strong><br>
                                <small class="file-path"><?php echo esc_html($threat['file']); ?></small>
                                <?php if (!empty($threat['line_number'])): ?>
                                    <br><small><?php _e('Line:', 'smoothbyte-security'); ?> <?php echo esc_html($threat['line_number']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <strong><?php echo esc_html($threat['signature_name'] ?? $threat['type']); ?></strong>
                            </td>
                            <td>
                                <?php echo esc_html($threat['description'] ?? $threat['type']); ?>
                                <?php if (!empty($threat['match'])): ?>
                                    <br><code class="threat-match"><?php echo esc_html(substr($threat['match'], 0, 50)) . '...'; ?></code>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button type="button" class="button button-small sbsec-quarantine-btn" 
                                        data-file="<?php echo esc_attr($threat['file']); ?>">
                                    <?php _e('Quarantine', 'smoothbyte-security'); ?>
                                </button>
                                <button type="button" class="button button-small button-link-delete sbsec-delete-btn" 
                                        data-file="<?php echo esc_attr($threat['file']); ?>">
                                    <?php _e('Delete', 'smoothbyte-security'); ?>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <!-- Warnings Tab -->
        <div id="warnings" class="tab-content">
            <?php if (empty($last_scan['warnings'])): ?>
                <div class="notice notice-info">
                    <p><?php _e('No warnings found.', 'smoothbyte-security'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Severity', 'smoothbyte-security'); ?></th>
                            <th><?php _e('File', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Issue', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Description', 'smoothbyte-security'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($last_scan['warnings'] as $warning): ?>
                        <tr>
                            <td>
                                <span class="severity-badge severity-<?php echo esc_attr($warning['severity']); ?>">
                                    <?php echo esc_html(strtoupper($warning['severity'])); ?>
                                </span>
                            </td>
                            <td>
                                <strong><?php echo esc_html(basename($warning['file'])); ?></strong><br>
                                <small class="file-path"><?php echo esc_html($warning['file']); ?></small>
                            </td>
                            <td><?php echo esc_html($warning['signature_name'] ?? $warning['type']); ?></td>
                            <td><?php echo esc_html($warning['description'] ?? 'Review recommended'); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <!-- Quarantine Tab -->
        <div id="quarantine" class="tab-content">
            <?php if (empty($quarantined)): ?>
                <div class="notice notice-info">
                    <p><?php _e('No files in quarantine.', 'smoothbyte-security'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('File', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Size', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Quarantined', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Reason', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Actions', 'smoothbyte-security'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($quarantined as $file): ?>
                        <tr>
                            <td>
                                <strong><?php echo esc_html(basename($file['original_path'])); ?></strong><br>
                                <small class="file-path"><?php echo esc_html($file['original_path']); ?></small>
                            </td>
                            <td><?php echo size_format($file['file_size']); ?></td>
                            <td><?php echo esc_html($file['quarantined_at']); ?></td>
                            <td><?php echo esc_html($file['reason']); ?></td>
                            <td>
                                <button type="button" class="button button-small sbsec-restore-btn" 
                                        data-id="<?php echo esc_attr($file['id']); ?>">
                                    <?php _e('Restore', 'smoothbyte-security'); ?>
                                </button>
                                <button type="button" class="button button-small button-link-delete sbsec-delete-quarantine-btn" 
                                        data-id="<?php echo esc_attr($file['id']); ?>">
                                    <?php _e('Delete Permanently', 'smoothbyte-security'); ?>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <!-- Scan History Tab -->
        <div id="history" class="tab-content">
            <?php if (empty($scan_history)): ?>
                <div class="notice notice-info">
                    <p><?php _e('No scan history available.', 'smoothbyte-security'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Date', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Files Scanned', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Threats', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Warnings', 'smoothbyte-security'); ?></th>
                            <th><?php _e('Duration', 'smoothbyte-security'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($scan_history as $scan): ?>
                        <tr>
                            <td><?php 
                                $scan_timestamp = strtotime($scan['scan_time']);
                                echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $scan_timestamp);
                            ?></td>
                            <td><?php echo esc_html($scan['files_scanned']); ?></td>
                            <td>
                                <span class="<?php echo $scan['threats_found'] > 0 ? 'danger' : 'success'; ?>">
                                    <?php echo esc_html($scan['threats_found']); ?>
                                </span>
                            </td>
                            <td><?php echo esc_html($scan['warnings_found']); ?></td>
                            <td><?php echo esc_html($scan['scan_duration']); ?>s</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
/* Scanner-specific styles */
.sbsec-status-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
    gap: 20px;
    margin: 24px 0;
}

.sbsec-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    padding: 24px;
    border: 2px solid var(--sbsec-border, #e2e8f0);
    border-radius: 12px;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    min-height: 120px;
    display: flex;
    flex-direction: column;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.sbsec-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, #2563eb, #3b82f6);
    transform: scaleX(0);
    transition: transform 0.3s ease;
}

.sbsec-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    border-color: #2563eb;
}

.sbsec-card:hover::before {
    transform: scaleX(1);
}

.sbsec-card h3 {
    margin: 0 0 12px 0;
    font-size: 13px;
    font-weight: 600;
    color: #64748b;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.sbsec-card .stat-value {
    font-size: clamp(10px, 2vw, 12px);
    font-weight: 600;
    color: #1e293b;
    line-height: 1.3;
    word-wrap: break-word;
    overflow-wrap: break-word;
    hyphens: auto;
}

/* For large numbers */
.sbsec-card .stat-value:not(:has(text)) {
    font-size: clamp(32px, 5vw, 48px);
    font-weight: 800;
    line-height: 1.2;
}

.sbsec-card .stat-value.success {
    background: linear-gradient(135deg, #10b981, #059669);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    font-size: clamp(32px, 5vw, 48px);
    font-weight: 800;
}

.sbsec-card .stat-value.danger {
    background: linear-gradient(135deg, #ef4444, #dc2626);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    font-size: clamp(32px, 5vw, 48px);
    font-weight: 800;
}

/* Responsive adjustments */
@media (max-width: 1200px) {
    .sbsec-status-cards {
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    }
    
    .sbsec-card .stat-value {
        font-size: clamp(9px, 2.5vw, 11px);
    }
}

@media (max-width: 782px) {
    .sbsec-status-cards {
        grid-template-columns: 1fr 1fr;
    }
    
    .sbsec-card {
        padding: 16px;
        min-height: 100px;
    }
    
    .sbsec-card .stat-value {
        font-size: clamp(9px, 3vw, 11px);
    }
    
    .sbsec-card .stat-value.success,
    .sbsec-card .stat-value.danger {
        font-size: clamp(28px, 7vw, 36px);
    }
}

@media (max-width: 480px) {
    .sbsec-status-cards {
        grid-template-columns: 1fr;
    }
}

.sbsec-scanner-controls {
    background: #fff;
    border: 1px solid #ccd0d4;
    padding: 20px;
    margin: 20px 0;
    display: flex;
    align-items: center;
    gap: 15px;
    flex-wrap: wrap;
}

.signature-info {
    margin-left: auto;
}

.sbsec-scan-progress {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-left: 4px solid #2271b1;
    padding: 20px;
    margin: 20px 0;
}

.progress-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.progress-bar-container {
    height: 30px;
    background: #f0f0f1;
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 10px;
}

.progress-bar {
    height: 100%;
    background: linear-gradient(90deg, #2271b1 0%, #135e96 100%);
    width: 0%;
    transition: width 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
}

.progress-info {
    display: flex;
    justify-content: space-between;
    font-size: 13px;
    color: #646970;
}

.sbsec-tabs {
    background: #fff;
    border: 1px solid #ccd0d4;
    margin: 20px 0;
}

.sbsec-tabs .nav-tab-wrapper {
    border-bottom: 1px solid #ccd0d4;
    margin: 0;
}

.sbsec-tabs .tab-content {
    display: none;
    padding: 20px;
}

.sbsec-tabs .tab-content.active {
    display: block;
}

.severity-badge {
    display: inline-block;
    padding: 3px 8px;
    border-radius: 3px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
}

.severity-critical {
    background: #d63638;
    color: white;
}

.severity-high {
    background: #f56e28;
    color: white;
}

.severity-medium {
    background: #dba617;
    color: white;
}

.severity-low {
    background: #007cba;
    color: white;
}

.file-path {
    color: #646970;
    word-break: break-all;
}

.threat-match {
    display: inline-block;
    background: #f0f0f1;
    padding: 2px 6px;
    margin-top: 5px;
    font-size: 11px;
    max-width: 100%;
    overflow: hidden;
    text-overflow: ellipsis;
}

.danger {
    color: #d63638 !important;
    font-weight: 600;
}

.success {
    color: #00a32a !important;
    font-weight: 600;
}
</style>

<script>
jQuery(document).ready(function($) {
    var scanInProgress = false;
    var progressInterval = null;
    
    // Tab switching (within scanner page only)
    $('.sbsec-tabs .nav-tab').on('click', function(e) {
        e.preventDefault();
        var target = $(this).attr('href');
        // Only handle hash-based tabs (like #threats, #warnings, etc.)
        if (target && target.indexOf('#') === 0) {
            $('.sbsec-tabs .nav-tab').removeClass('nav-tab-active');
            $(this).addClass('nav-tab-active');
            $('.sbsec-tabs .tab-content').removeClass('active');
            $(target).addClass('active');
        }
    });
    
    // Run scan
    $('#sbsec-run-scan').on('click', function() {
        if (scanInProgress) return;
        
        if (!confirm('<?php _e('Start a full malware scan? This may take several minutes.', 'smoothbyte-security'); ?>')) {
            return;
        }
        
        scanInProgress = true;
        $('#scan-progress').fadeIn();
        $(this).prop('disabled', true).text('<?php _e('Scanning...', 'smoothbyte-security'); ?>');
        
        // Start progress polling
        progressInterval = setInterval(checkProgress, 2000);
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_run_scan',
                nonce: sbsecAdmin.nonce
            },
            success: function(response) {
                clearInterval(progressInterval);
                scanInProgress = false;
                
                if (response.success) {
                    alert('<?php _e('Scan completed successfully!', 'smoothbyte-security'); ?>');
                    location.reload();
                } else {
                    alert('<?php _e('Scan failed:', 'smoothbyte-security'); ?> ' + (response.data || 'Unknown error'));
                }
            },
            error: function() {
                clearInterval(progressInterval);
                scanInProgress = false;
                alert('<?php _e('Scan failed. Please try again.', 'smoothbyte-security'); ?>');
                $('#sbsec-run-scan').prop('disabled', false).html('<span class="dashicons dashicons-search"></span><?php _e('Run Full Scan', 'smoothbyte-security'); ?>');
            }
        });
    });
    
    // Check scan progress
    function checkProgress() {
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_get_scan_progress',
                nonce: sbsecAdmin.nonce
            },
            success: function(response) {
                if (response.success && response.data) {
                    var progress = response.data;
                    $('#progress-bar').css('width', progress.progress_percent + '%').text(progress.progress_percent + '%');
                    $('#progress-status').text(progress.current_file || 'Scanning...');
                    $('#progress-details').text('Files scanned: ' + (progress.files_scanned || 0) + ' | Threats: ' + (progress.threats_found || 0));
                }
            }
        });
    }
    
    // Update signatures
    $('#sbsec-update-signatures').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update dashicons-spin"></span><?php _e('Updating...', 'smoothbyte-security'); ?>');
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_update_signatures',
                nonce: sbsecAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    alert(response.data.message + '\n' + 'Patterns: ' + response.data.count);
                    location.reload();
                } else {
                    alert('<?php _e('Failed to update signatures', 'smoothbyte-security'); ?>');
                }
            },
            complete: function() {
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span><?php _e('Update Signatures', 'smoothbyte-security'); ?>');
            }
        });
    });
    
    // Quarantine file
    $('.sbsec-quarantine-btn').on('click', function() {
        if (!confirm('<?php _e('Quarantine this file?', 'smoothbyte-security'); ?>')) return;
        
        var $btn = $(this);
        var filePath = $btn.data('file');
        
        $btn.prop('disabled', true);
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_quarantine_file',
                nonce: sbsecAdmin.nonce,
                file_path: filePath
            },
            success: function(response) {
                alert(response.success ? response.data.message : response.data);
                if (response.success) location.reload();
            },
            complete: function() {
                $btn.prop('disabled', false);
            }
        });
    });
    
    // Restore file
    $('.sbsec-restore-btn').on('click', function() {
        if (!confirm('<?php _e('Restore this file from quarantine?', 'smoothbyte-security'); ?>')) return;
        
        var $btn = $(this);
        var id = $btn.data('id');
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_restore_file',
                nonce: sbsecAdmin.nonce,
                quarantine_id: id
            },
            success: function(response) {
                alert(response.success ? response.data.message : response.data);
                if (response.success) location.reload();
            }
        });
    });
    
    // Delete file
    $('.sbsec-delete-btn').on('click', function() {
        if (!confirm('<?php _e('Permanently delete this file? This cannot be undone!', 'smoothbyte-security'); ?>')) return;
        
        var $btn = $(this);
        var filePath = $btn.data('file');
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_delete_threat',
                nonce: sbsecAdmin.nonce,
                file_path: filePath
            },
            success: function(response) {
                alert(response.success ? response.data.message : response.data);
                if (response.success) location.reload();
            }
        });
    });
    
    // Delete quarantined file
    $('.sbsec-delete-quarantine-btn').on('click', function() {
        if (!confirm('<?php _e('Permanently delete this quarantined file? This cannot be undone!', 'smoothbyte-security'); ?>')) return;
        
        var $btn = $(this);
        var id = $btn.data('id');
        
        $.ajax({
            url: sbsecAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'sbsec_delete_threat',
                nonce: sbsecAdmin.nonce,
                action_type: 'quarantine',
                quarantine_id: id
            },
            success: function(response) {
                alert(response.success ? response.data.message : response.data);
                if (response.success) location.reload();
            }
        });
    });
});
</script>
