<?php
if (!defined('ABSPATH')) exit;

class SBSEC_Admin_Settings {
    private $options;
    
    public function __construct($options) {
        $this->options = $options;
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_sbsec_test_headers', array($this, 'ajax_test_headers'));
        add_action('wp_ajax_sbsec_request_access', array($this, 'ajax_request_access'));
        add_action('wp_ajax_sbsec_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_sbsec_check_updates', array($this, 'ajax_check_updates'));
        add_action('wp_ajax_sbsec_report_health', array($this, 'ajax_report_health'));
        add_action('wp_ajax_sbsec_update_signatures', array($this, 'ajax_update_signatures'));
        add_action('wp_ajax_sbsec_run_scan', array($this, 'ajax_run_scan'));
        add_action('wp_ajax_sbsec_get_scan_progress', array($this, 'ajax_get_scan_progress'));
        add_action('wp_ajax_sbsec_quarantine_file', array($this, 'ajax_quarantine_file'));
        add_action('wp_ajax_sbsec_restore_file', array($this, 'ajax_restore_file'));
        add_action('wp_ajax_sbsec_delete_threat', array($this, 'ajax_delete_threat'));
        add_action('wp_ajax_sbsec_install_update', array($this, 'ajax_install_update'));
    }
    
    public function add_admin_menu() {
        add_menu_page('SmoothByte Security', 'Security', 'manage_options', 'smoothbyte-security', 
            array($this, 'render_settings_page'), 'dashicons-shield', 75);
    }
    
    /**
     * Check if plugin update is available
     */
    private function check_update_available() {
        $current_version = SBSEC_VERSION;
        $update_plugins = get_site_transient('update_plugins');
        
        if (isset($update_plugins->response[SBSEC_PLUGIN_BASENAME])) {
            $update = $update_plugins->response[SBSEC_PLUGIN_BASENAME];
            if (version_compare($update->new_version, $current_version, '>')) {
                return $update;
            }
        }
        
        return false;
    }
    
    public function register_settings() {
        register_setting('sbsec_options_group', 'sbsec_options');
        register_setting('sbsec_options', 'sbsec_options');
    }
    
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_smoothbyte-security') return;
        wp_enqueue_style('sbsec-admin', SBSEC_PLUGIN_URL . 'assets/css/admin.css', array(), SBSEC_VERSION);
        wp_enqueue_script('sbsec-admin', SBSEC_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), SBSEC_VERSION, true);
        wp_localize_script('sbsec-admin', 'sbsecAdmin', array('ajaxurl' => admin_url('admin-ajax.php'), 'nonce' => wp_create_nonce('sbsec_admin')));
    }
    
    public function render_settings_page() {
        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';
        echo '<div class="wrap"><h1><span class="dashicons dashicons-shield"></span> SmoothByte Security Pro</h1>';
        
        // Check for available updates
        $update = $this->check_update_available();
        if ($update) {
            echo '<div class="notice notice-warning sbsec-update-notice" style="display: flex; align-items: center; justify-content: space-between; padding: 12px 15px; margin: 15px 0; border-left: 4px solid #f59e0b;">';
            echo '<div>';
            echo '<p style="margin: 0; font-weight: 600;"><span class="dashicons dashicons-info" style="color: #f59e0b;"></span> ';
            echo sprintf(__('Update Available: Version %s is ready to install.', 'smoothbyte-security'), esc_html($update->new_version));
            echo '</p>';
            echo '<p style="margin: 5px 0 0 0; font-size: 13px; color: #666;">Current version: ' . SBSEC_VERSION . '</p>';
            echo '</div>';
            echo '<button type="button" class="button button-primary" id="sbsec-install-update" data-version="' . esc_attr($update->new_version) . '">';
            echo '<span class="dashicons dashicons-download" style="margin-top: 3px;"></span> Install Update Now';
            echo '</button>';
            echo '</div>';
        }
        
        echo '<h2 class="nav-tab-wrapper">';
        $tabs = array(
            'dashboard' => 'Dashboard', 
            'firewall' => 'Firewall', 
            'headers' => 'Security Headers',
            'scanner' => 'Scanner', 
            'logs' => 'Logs', 
            'updates' => 'Updates'
        );
        foreach ($tabs as $key => $label) {
            $active = $tab === $key ? 'nav-tab-active' : '';
            echo '<a href="?page=smoothbyte-security&tab=' . $key . '" class="nav-tab ' . $active . '">' . $label . '</a>';
        }
        echo '</h2>';
        
        switch ($tab) {
            case 'firewall': $this->render_firewall(); break;
            case 'headers': $this->render_security_headers(); break;
            case 'scanner': $this->render_scanner(); break;
            case 'logs': $this->render_logs(); break;
            case 'updates': $this->render_updates(); break;
            default: $this->render_dashboard(); break;
        }
        echo '</div>';
    }
    
    private function render_dashboard() {
        $stats = SBSEC_Security_Logs::get_statistics(7);
        echo '<h2>Security Dashboard</h2>';
        echo '<div class="sbsec-stats-grid">';
        echo '<div class="sbsec-stat-box"><h3>Events (7d)</h3><div class="stat-number">' . ($stats['total_events'] ?? 0) . '</div></div>';
        echo '<div class="sbsec-stat-box"><h3>Blocked IPs</h3><div class="stat-number">' . count(SBSEC_Firewall::get_blocked_ips()) . '</div></div>';
        echo '<div class="sbsec-stat-box"><h3>Last Scan</h3><div class="stat-number">Never</div></div>';
        echo '</div>';
    }
    
    private function render_firewall() {
        echo '<h2>Firewall Settings</h2>';
        echo '<form method="post" action="options.php">';
        settings_fields('sbsec_options_group');
        echo '<table class="form-table">';
        echo '<tr><th>Enable Firewall</th><td><input type="checkbox" name="sbsec_options[firewall]" value="1" ' . checked($this->options['firewall'] ?? false, true, false) . '></td></tr>';
        echo '<tr><th>Rate Limit (req/min)</th><td><input type="number" name="sbsec_options[firewall_rate_limit]" value="' . esc_attr($this->options['firewall_rate_limit'] ?? 60) . '"></td></tr>';
        echo '</table>';
        submit_button();
        echo '</form>';
    }
    
    private function render_scanner() {
        include SBSEC_PLUGIN_DIR . 'admin/tabs/scanner.php';
    }
    
    private function render_logs() {
        echo '<h2>Security Logs</h2>';
        $logs = SBSEC_Security_Logs::get_logs(array(), 50);
        echo '<table class="wp-list-table widefat">';
        echo '<thead><tr><th>Time</th><th>Event</th><th>IP</th><th>Message</th></tr></thead><tbody>';
        foreach ($logs as $log) {
            echo '<tr><td>' . esc_html($log->timestamp) . '</td>';
            echo '<td>' . esc_html($log->event_type) . '</td>';
            echo '<td>' . esc_html($log->ip_address) . '</td>';
            echo '<td>' . esc_html($log->message) . '</td></tr>';
        }
        echo '</tbody></table>';
    }
    
    private function render_security_headers() {
        include SBSEC_PLUGIN_DIR . 'admin/tabs/security-headers.php';
    }
    
    public function ajax_test_headers() {
        check_ajax_referer('sbsec_test_headers', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Make a HEAD request to test headers
        $response = wp_remote_head(home_url(), array(
            'timeout' => 10,
            'sslverify' => false,
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $headers = wp_remote_retrieve_headers($response);
        
        // Check each security header
        $results = array();
        
        $header_tests = array(
            'xframe' => array('name' => 'X-Frame-Options', 'key' => 'x-frame-options'),
            'xss' => array('name' => 'X-XSS-Protection', 'key' => 'x-xss-protection'),
            'content_type' => array('name' => 'X-Content-Type-Options', 'key' => 'x-content-type-options'),
            'hsts' => array('name' => 'Strict-Transport-Security', 'key' => 'strict-transport-security'),
            'referrer' => array('name' => 'Referrer-Policy', 'key' => 'referrer-policy'),
            'permissions' => array('name' => 'Permissions-Policy', 'key' => 'permissions-policy'),
        );
        
        foreach ($header_tests as $key => $test) {
            $header_value = '';
            $status = 'inactive';
            
            // Check if header exists (try both formats)
            if (isset($headers[$test['key']])) {
                $header_value = $headers[$test['key']];
                $status = 'active';
            } elseif (isset($headers[str_replace('-', '_', $test['key'])])) {
                $header_value = $headers[str_replace('-', '_', $test['key'])];
                $status = 'active';
            }
            
            $results[$key] = array(
                'name' => $test['name'],
                'status' => $status,
                'value' => $header_value,
            );
        }
        
        wp_send_json_success(array('headers' => $results));
    }
    
    public function ajax_request_access() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        
        if (empty($server_url)) {
            wp_send_json_error('Server URL is required');
        }
        
        // Automatic registration (free service, no master key needed)
        $response = wp_remote_post($server_url . '/api.php/request-access', array(
            'body' => json_encode(array(
                'site_url' => get_site_url(),
                'site_name' => get_bloginfo('name'),
            )),
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!empty($data['success']) && !empty($data['api_key'])) {
            update_option('sbsec_update_server_url', $server_url);
            update_option('sbsec_update_api_key', $data['api_key']);
            update_option('sbsec_site_id', $data['site_id'] ?? '');
            
            wp_send_json_success(array(
                'message' => '✓ Successfully registered! ' . ($data['message'] ?? 'API key received.'),
            ));
        } else {
            wp_send_json_error($data['error'] ?? 'Failed to register site');
        }
    }
    
    public function ajax_test_connection() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $server_url = get_option('sbsec_update_server_url', '');
        $api_key = get_option('sbsec_update_api_key', '');
        
        if (empty($server_url) || empty($api_key)) {
            wp_send_json_error('Not configured');
        }
        
        // Test connection to deployment server
        $response = wp_remote_get($server_url . '/api.php/info', array(
            'headers' => array(
                'X-API-Key' => $api_key,
            ),
            'timeout' => 10,
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!empty($data['success'])) {
            wp_send_json_success(array(
                'version' => $data['app_version'] ?? 'Unknown',
            ));
        } else {
            wp_send_json_error($data['error'] ?? 'Connection test failed');
        }
    }
    
    public function ajax_check_updates() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Clear the update transient to force WordPress to check for updates
        delete_site_transient('update_plugins');
        
        // Trigger update check by refreshing transient
        wp_update_plugins();
        
        wp_send_json_success(array(
            'message' => 'Update check completed. Go to WordPress Dashboard → Updates to see results.',
        ));
    }
    
    public function ajax_report_health() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Trigger health report
        $updater = SBSEC_Auto_Updater::get_instance();
        if ($updater) {
            $updater->report_health();
            wp_send_json_success(array(
                'message' => 'Health report sent successfully to deployment server.',
            ));
        } else {
            wp_send_json_error('Auto-updater not available');
        }
    }
    
    public function ajax_update_signatures() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $server_url = get_option('sbsec_update_server_url', 'https://wpdeploy.smoothbyteit.dev');
        $api_key = get_option('sbsec_update_api_key', '');
        
        if (empty($server_url) || empty($api_key)) {
            wp_send_json_error('Deployment server not configured. Please register site first.');
            return;
        }
        
        // Request malware signatures from deployment server (correct endpoint)
        $url = rtrim($server_url, '/') . '/api.php/malware-signatures';
        
        // Debug: Log request details
        SBSEC_Security_Logs::log('signature_update', 'info', '', 0, 'Requesting signatures from: ' . $url . ' | API Key: ' . substr($api_key, 0, 8) . '...');
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'X-API-Key' => $api_key,
            ),
            'sslverify' => false, // For development
        ));
        
        if (is_wp_error($response)) {
            $error_msg = $response->get_error_message();
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Failed to connect: ' . $error_msg);
            wp_send_json_error('Failed to connect to server: ' . $error_msg);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // Debug: Log response
        SBSEC_Security_Logs::log('signature_update', 'info', '', 0, 'Response: HTTP ' . $status_code . ' | Body length: ' . strlen($body) . ' bytes');
        
        if ($status_code !== 200) {
            $error_msg = $data['error'] ?? 'HTTP ' . $status_code;
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, 'Failed: ' . $error_msg . ' | Response: ' . substr($body, 0, 500));
            wp_send_json_error('Server error: ' . $error_msg);
            return;
        }
        
        if (empty($data['success'])) {
            SBSEC_Security_Logs::log('signature_update', 'error', '', 0, $data['error'] ?? 'Unknown error');
            wp_send_json_error($data['error'] ?? 'Failed to download signatures');
            return;
        }
        
        // Store signatures
        if (!empty($data['signatures'])) {
            set_transient('sbsec_malware_signatures', $data['signatures'], DAY_IN_SECONDS);
            update_option('sbsec_signatures_version', $data['version'] ?? '1.0');
            update_option('sbsec_signatures_updated', current_time('mysql'));
            
            $count = $this->count_signature_patterns($data['signatures']);
            
            SBSEC_Security_Logs::log('signature_update', 'info', '', 0, 'Malware signatures updated successfully (' . $count . ' patterns)');
            
            wp_send_json_success(array(
                'message' => 'Signatures updated successfully! Downloaded ' . $count . ' malware patterns.',
                'version' => $data['version'] ?? '1.0',
                'count' => $count,
                'updated' => current_time('mysql')
            ));
        } else {
            wp_send_json_error('No signatures received from server');
        }
    }
    
    private function count_signature_patterns($signatures) {
        $count = 0;
        foreach ($signatures as $category => $patterns) {
            if (is_array($patterns)) {
                $count += count($patterns);
            }
        }
        return $count;
    }
    
    public function ajax_run_scan() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Ensure scan options are enabled
        $scan_options = array_merge($this->options, array(
            'scan_wp_core' => true,
            'scan_plugins' => true,
            'scan_themes' => true,
            'scan_uploads' => true,
        ));
        
        $scanner = new SBSEC_Malware_Scanner($scan_options);
        $result = $scanner->run_scan();
        
        if ($result) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error('Scan failed');
        }
    }
    
    public function ajax_get_scan_progress() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $progress = get_transient('sbsec_scan_progress');
        wp_send_json_success($progress ?: array('progress_percent' => 0));
    }
    
    public function ajax_quarantine_file() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
        
        if (empty($file_path)) {
            wp_send_json_error('No file specified');
        }
        
        $result = SBSEC_Malware_Scanner::quarantine_file($file_path, 'Manual quarantine from admin');
        
        if ($result) {
            wp_send_json_success(array('message' => 'File quarantined successfully'));
        } else {
            wp_send_json_error('Failed to quarantine file');
        }
    }
    
    public function ajax_restore_file() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $id = isset($_POST['quarantine_id']) ? intval($_POST['quarantine_id']) : 0;
        
        if (!$id) {
            wp_send_json_error('No file specified');
        }
        
        $result = SBSEC_Malware_Scanner::restore_file($id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'File restored successfully'));
        } else {
            wp_send_json_error('Failed to restore file');
        }
    }
    
    public function ajax_delete_threat() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
        
        if (empty($file_path)) {
            wp_send_json_error('No file specified');
        }
        
        if (file_exists($file_path) && @unlink($file_path)) {
            SBSEC_Security_Logs::log('file_deleted', 'high', 'Threat file deleted: ' . $file_path);
            wp_send_json_success(array('message' => 'File deleted successfully'));
        } else {
            wp_send_json_error('Failed to delete file');
        }
    }
    
    public function ajax_install_update() {
        check_ajax_referer('sbsec_admin', 'nonce');
        
        if (!current_user_can('update_plugins')) {
            wp_send_json_error('You do not have permission to update plugins');
        }
        
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        
        // Create upgrader instance
        $skin = new WP_Ajax_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        
        // Perform the upgrade
        $result = $upgrader->upgrade(SBSEC_PLUGIN_BASENAME);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } elseif ($result === false) {
            wp_send_json_error('Update failed. Please try again or update manually from the Plugins page.');
        } else {
            // Log the update
            SBSEC_Security_Logs::log('plugin_updated', 'info', '', 0, 'Plugin updated successfully to version ' . SBSEC_VERSION);
            
            wp_send_json_success(array(
                'message' => 'Plugin updated successfully! The page will reload.',
                'reload' => true
            ));
        }
    }
    
    private function render_updates() {
        $server_url = get_option('sbsec_update_server_url', 'https://wpdeploy.smoothbyteit.dev');
        $api_key = get_option('sbsec_update_api_key', '');
        $configured = !empty($server_url) && !empty($api_key);
        
        // Check for available updates
        $update = $this->check_update_available();
        
        echo '<h2>Automatic Updates</h2>';
        echo '<p>Connect to SmoothByte deployment server for automatic updates.</p>';
        
        // Show update available notice
        if ($update && $configured) {
            echo '<div class="notice notice-warning inline" style="border-left-color: #f59e0b; padding: 15px; margin: 20px 0; display: flex; align-items: center; justify-content: space-between;">';
            echo '<div>';
            echo '<p style="margin: 0; font-weight: 600;"><span class="dashicons dashicons-update" style="color: #f59e0b;"></span> ';
            echo sprintf(__('New Version Available: %s', 'smoothbyte-security'), '<strong>' . esc_html($update->new_version) . '</strong>');
            echo '</p>';
            echo '<p style="margin: 5px 0 0 0; font-size: 13px;">You are currently running version ' . SBSEC_VERSION . '</p>';
            echo '</div>';
            echo '<button type="button" class="button button-primary button-large" id="sbsec-install-update-tab" data-version="' . esc_attr($update->new_version) . '" style="margin-left: 15px;">';
            echo '<span class="dashicons dashicons-download"></span> Install Update Now';
            echo '</button>';
            echo '</div>';
        }
        
        if (!$configured) {
            echo '<div class="notice notice-info inline"><p>';
            echo '<strong>🔑 Free Automatic Registration</strong><br>';
            echo 'Click the button below to automatically request access to the deployment server.<br>';
            echo 'Your site will be registered and receive an API key instantly - no manual configuration needed!';
            echo '</p></div>';
        }
        
        echo '<table class="form-table">';
        echo '<tr><th>Server URL</th><td><input type="text" id="sbsec-server-url" class="regular-text" value="' . esc_attr($server_url) . '" readonly /></td></tr>';
        echo '<tr><th>Status</th><td>' . ($configured ? '<span style="color:green;">✓ Connected</span>' : '<span style="color:orange;">Not Configured</span>') . '</td></tr>';
        
        if ($configured) {
            echo '<tr><th>Site ID</th><td>' . get_option('sbsec_site_id', 'N/A') . '</td></tr>';
        }
        
        echo '<tr><th>Current Version</th><td><strong>' . SBSEC_VERSION . '</strong></td></tr>';
        
        if ($update && $configured) {
            echo '<tr><th>Available Version</th><td><strong style="color: #f59e0b;">' . esc_html($update->new_version) . '</strong> <span style="color: #10b981;">⬆ Update Available</span></td></tr>';
        }
        
        echo '</table>';
        echo '<p><button type="button" class="button button-primary button-large" id="sbsec-request-access">';
        echo '<span class="dashicons dashicons-admin-network"></span> ' . ($configured ? 'Re-register Site' : 'Request Access (Free)');
        echo '</button>';
        if ($configured) {
            echo ' <button type="button" class="button button-secondary" id="sbsec-test-connection">Test Connection</button>';
            echo ' <button type="button" class="button button-secondary" id="sbsec-check-updates">Check for Updates Now</button>';
            echo ' <button type="button" class="button button-secondary" id="sbsec-report-health">Report Health to Server</button>';
        }
        echo '</p><div id="sbsec-update-message"></div>';
    }
}
