<?php
/**
 * Plugin Name: SEO & AEO Optimizer Pro
 * Plugin URI: https://smoothbyteit.dev
 * Description: Comprehensive SEO and Answer Engine Optimization (AEO) with Schema.org markup, AI crawler support, and advanced meta tags. Optimized for ChatGPT, Claude, Perplexity, and traditional search engines.
 * Version: 2.7.0
 * Author: SmoothByte IT
 * Author URI: https://smoothbyteit.dev
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: seo-aeo-optimizer
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('SEO_AEO_VERSION', '2.7.0');
define('SEO_AEO_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SEO_AEO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SEO_AEO_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main Plugin Class
 */
class SEO_AEO_Optimizer {

    /**
     * Single instance of the class
     */
    private static $instance = null;

    /**
     * Plugin options
     */
    private $options;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required dependencies
     */
    private function load_dependencies() {
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-schema-generator.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-meta-tags.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-robots-optimizer.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-local-seo.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-advanced-schema.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-technical-seo.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-sitemap-generator.php';
        require_once SEO_AEO_PLUGIN_DIR . 'includes/class-auto-updater.php';
        
        // Load auction extension if auction plugin is active
        if (file_exists(SEO_AEO_PLUGIN_DIR . 'includes/class-auction-schema.php')) {
            require_once SEO_AEO_PLUGIN_DIR . 'includes/class-auction-schema.php';
        }
        
        if (is_admin()) {
            require_once SEO_AEO_PLUGIN_DIR . 'admin/class-admin-settings.php';
        }
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // Activation & Deactivation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Initialize components
        add_action('plugins_loaded', array($this, 'init'));
        
        // Add settings link on plugins page
        add_filter('plugin_action_links_' . SEO_AEO_PLUGIN_BASENAME, array($this, 'add_settings_link'));
    }

    /**
     * Initialize plugin components
     */
    public function init() {
        // Load options
        $this->options = get_option('seo_aeo_options', $this->get_default_options());

        // Initialize schema generator
        if ($this->is_enabled('schema')) {
            new SEO_AEO_Schema_Generator($this->options);
        }

        // Initialize meta tags
        if ($this->is_enabled('meta_tags')) {
            new SEO_AEO_Meta_Tags($this->options);
        }

        // Initialize robots optimizer
        if ($this->is_enabled('robots')) {
            new SEO_AEO_Robots_Optimizer($this->options);
        }

        // Initialize local SEO (GEO)
        if ($this->is_enabled('local_seo_enabled')) {
            new SEO_AEO_Local_SEO($this->options);
        }

        // Initialize advanced schema
        new SEO_AEO_Advanced_Schema($this->options);

        // Initialize technical SEO
        new SEO_AEO_Technical_SEO($this->options);
        
        // Initialize sitemap generator
        if ($this->is_enabled('sitemap_enabled')) {
            new SEO_AEO_Sitemap_Generator($this->options);
        }
        
        // Initialize auto-updater
        new SEO_AEO_Auto_Updater(SEO_AEO_PLUGIN_BASENAME, SEO_AEO_VERSION);

        // Initialize admin panel
        if (is_admin()) {
            new SEO_AEO_Admin_Settings($this->options);
            
            // Add AJAX handlers
            add_action('wp_ajax_seo_aeo_import_settings', array($this, 'ajax_import_settings'));
            add_action('wp_ajax_seo_aeo_test_update_connection', array($this, 'ajax_test_update_connection'));
            add_action('wp_ajax_seo_aeo_check_updates_now', array($this, 'ajax_check_updates_now'));
            add_action('wp_ajax_seo_aeo_report_health_now', array($this, 'ajax_report_health_now'));
            add_action('wp_ajax_seo_aeo_save_update_settings', array($this, 'ajax_save_update_settings'));
            add_action('wp_ajax_seo_aeo_register_site', array($this, 'ajax_register_site'));
            add_action('wp_ajax_seo_aeo_request_access', array($this, 'ajax_request_access'));
            add_action('wp_ajax_seo_aeo_clear_registration', array($this, 'ajax_clear_registration'));
            add_action('wp_ajax_seo_aeo_clear_sitemap_cache', array($this, 'ajax_clear_sitemap_cache'));
            add_action('wp_ajax_seo_aeo_ping_sitemap', array($this, 'ajax_ping_sitemap'));
        }
    }
    
    /**
     * AJAX handler for importing settings
     */
    public function ajax_import_settings() {
        check_ajax_referer('seo_aeo_import', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $settings = isset($_POST['settings']) ? json_decode(stripslashes($_POST['settings']), true) : null;
        
        if ($settings && is_array($settings)) {
            update_option('seo_aeo_options', $settings);
            wp_send_json_success('Settings imported successfully');
        } else {
            wp_send_json_error('Invalid settings data');
        }
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        if (!get_option('seo_aeo_options')) {
            add_option('seo_aeo_options', $this->get_default_options());
        }

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Clear scheduled compatibility checks
        $timestamp = wp_next_scheduled('seo_aeo_daily_compat_check');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'seo_aeo_daily_compat_check');
        }
    }

    /**
     * Get default plugin options
     */
    private function get_default_options() {
        return array(
            // General Settings
            'enabled' => true,
            
            // Sitemap Settings
            'sitemap_enabled' => true,
            'sitemap_post_types' => array('post', 'page'),
            'post_types' => array('post', 'page', 'product'),
            
            // Schema Settings (all enabled for maximum SEO/AEO coverage)
            'schema' => true,
            'schema_organization' => true,
            'schema_product' => true,
            'schema_breadcrumb' => true,
            'schema_faq' => true,
            'schema_article' => true,
            'schema_event' => true,       // Enabled for auction events
            'schema_review' => true,      // Enabled for ratings/reviews
            'schema_video' => true,       // Enabled for video content
            'schema_howto' => true,       // Enabled for AEO optimization
            'schema_speakable' => true,   // Enabled for voice search
            
            // Organization Details
            'org_name' => get_bloginfo('name'),
            'org_url' => home_url('/'),
            'org_logo' => '',
            'org_description' => get_bloginfo('description'),
            'org_social_links' => array(),
            
            // Meta Tags
            'meta_tags' => true,
            'og_tags' => true,
            'twitter_cards' => true,
            
            // AI Crawlers
            'robots' => true,
            'allow_gptbot' => true,
            'allow_ccbot' => true,
            'allow_anthropic' => true,
            'allow_perplexity' => true,
            'allow_youbot' => true,
            
            // FAQ Settings
            'faq_enabled' => true,
            'faq_items' => array(),
            
            // Local SEO / GEO (enabled by default for local business visibility)
            'local_seo_enabled' => true,
            'local_business_type' => 'LocalBusiness',
            'local_address' => '',
            'local_street_address' => '',
            'local_city' => '',
            'local_state' => '',
            'local_postal_code' => '',
            'local_country' => '',
            'local_latitude' => '',
            'local_longitude' => '',
            'local_phone' => '',
            'local_email' => '',
            'local_opening_hours' => array(),
            'local_price_range' => '',
            'local_service_areas' => array(),
            'local_contact_page' => 0,
            
            // Technical SEO
            'canonical_enabled' => true,
            'noindex_archives' => false,
            'noindex_search' => true,
            'noindex_paginated' => false,
            'max_image_preview' => 'large',
            'max_snippet' => '-1',
            'max_video_preview' => '-1',
            'image_optimization' => true,
            'dns_prefetch' => true,
            'resource_hints' => array(),
            
            // Advanced
            'debug_mode' => false,
            'cache_schema' => true,
        );
    }

    /**
     * Check if a feature is enabled
     */
    private function is_enabled($feature) {
        return !empty($this->options[$feature]);
    }

    /**
     * Add settings link on plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="admin.php?page=seo-aeo-optimizer">' . __('Settings', 'seo-aeo-optimizer') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * AJAX handler for testing update connection
     */
    public function ajax_test_update_connection() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
        
        if (empty($server_url) || empty($api_key)) {
            wp_send_json_error('Server URL and API Key are required');
            return;
        }
        
        // Test connection
        $url = rtrim($server_url, '/') . '/api.php/info';
        $response = wp_remote_get($url, array(
            'headers' => array('X-API-Key' => $api_key),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
            return;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['app_version'])) {
            wp_send_json_success(array(
                'message' => 'Connection successful!',
                'version' => $body['app_version']
            ));
        } else {
            wp_send_json_error('Invalid response from server');
        }
    }
    
    /**
     * AJAX handler for checking updates now
     */
    public function ajax_check_updates_now() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        delete_site_transient('update_plugins');
        wp_update_plugins();
        
        $updates = get_site_transient('update_plugins');
        $has_update = isset($updates->response[SEO_AEO_PLUGIN_BASENAME]);
        
        if ($has_update) {
            $update_data = $updates->response[SEO_AEO_PLUGIN_BASENAME];
            wp_send_json_success(array(
                'has_update' => true,
                'new_version' => $update_data->new_version,
                'message' => 'Update available: version ' . $update_data->new_version
            ));
        } else {
            wp_send_json_success(array(
                'has_update' => false,
                'message' => 'Plugin is up to date'
            ));
        }
    }
    
    /**
     * AJAX handler for reporting health to server
     */
    public function ajax_report_health_now() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Trigger health report
        $updater = new SEO_AEO_Auto_Updater(SEO_AEO_PLUGIN_BASENAME, SEO_AEO_VERSION);
        $updater->report_health();
        
        // Return the data that was sent
        wp_send_json_success(array(
            'message' => 'Health report sent successfully',
            'wp_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'plugin_version' => SEO_AEO_VERSION
        ));
    }
    
    /**
     * AJAX handler for saving update settings
     */
    public function ajax_save_update_settings() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
        $auto_update = isset($_POST['auto_update']) ? (bool) $_POST['auto_update'] : false;
        
        update_option('seo_aeo_update_server_url', $server_url);
        update_option('seo_aeo_update_api_key', $api_key);
        update_option('seo_aeo_auto_update', $auto_update);
        
        wp_send_json_success(array(
            'message' => 'Update settings saved successfully!'
        ));
    }
    
    /**
     * AJAX handler for registering site
     */
    public function ajax_register_site() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        $master_key = isset($_POST['master_key']) ? sanitize_text_field($_POST['master_key']) : '';
        
        if (empty($server_url) || empty($master_key)) {
            wp_send_json_error('Server URL and Master Key are required');
            return;
        }
        
        // Save server URL first
        update_option('seo_aeo_update_server_url', $server_url);
        
        // Create updater instance and register
        $updater = new SEO_AEO_Auto_Updater(SEO_AEO_PLUGIN_BASENAME, SEO_AEO_VERSION);
        $result = $updater->register_site($master_key);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX handler for automatic access request (no master key needed)
     */
    public function ajax_request_access() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        
        if (empty($server_url)) {
            wp_send_json_error('Deployment Server URL is required');
            return;
        }
        
        // Save server URL first
        update_option('seo_aeo_update_server_url', $server_url);
        
        // Create updater instance and request access
        $updater = new SEO_AEO_Auto_Updater(SEO_AEO_PLUGIN_BASENAME, SEO_AEO_VERSION);
        $result = $updater->request_access();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX handler for clearing registration
     */
    public function ajax_clear_registration() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Clear the API key
        delete_option('seo_aeo_update_api_key');
        
        wp_send_json_success(array(
            'message' => 'Registration cleared successfully'
        ));
    }
    
    /**
     * AJAX handler for clearing sitemap cache
     */
    public function ajax_clear_sitemap_cache() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Clear sitemap cache
        global $wpdb;
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE '_transient_seo_aeo_sitemap_cache%' 
            OR option_name LIKE '_transient_timeout_seo_aeo_sitemap_cache%'"
        );
        
        wp_send_json_success(array(
            'message' => 'Sitemap cache cleared successfully'
        ));
    }
    
    /**
     * AJAX handler for pinging search engines
     */
    public function ajax_ping_sitemap() {
        check_ajax_referer('seo_aeo_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        $sitemap_url = urlencode(home_url('/sitemap.xml'));
        
        // Ping Google
        wp_remote_get("https://www.google.com/ping?sitemap={$sitemap_url}", array('timeout' => 5));
        
        // Ping Bing
        wp_remote_get("https://www.bing.com/ping?sitemap={$sitemap_url}", array('timeout' => 5));
        
        wp_send_json_success(array(
            'message' => 'Search engines pinged successfully'
        ));
    }
}

// Initialize the plugin
function seo_aeo_optimizer() {
    return SEO_AEO_Optimizer::get_instance();
}

// Start the plugin
seo_aeo_optimizer();
