<?php
/**
 * Technical SEO Class
 * Canonical URLs, Meta Robots, Image Optimization, etc.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Technical_SEO {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        // Canonical URLs
        if ($this->options['canonical_enabled'] ?? true) {
            add_action('wp_head', array($this, 'add_canonical_url'), 1);
        }

        // Meta Robots
        add_action('wp_head', array($this, 'add_meta_robots'), 2);

        // Image optimization hints
        if ($this->options['image_optimization'] ?? true) {
            add_filter('wp_get_attachment_image_attributes', array($this, 'optimize_image_attributes'), 10, 2);
        }

        // Preconnect/DNS-Prefetch
        if ($this->options['dns_prefetch'] ?? true) {
            add_action('wp_head', array($this, 'add_resource_hints'), 1);
        }

        // Remove unnecessary meta tags
        remove_action('wp_head', 'wp_generator');
        remove_action('wp_head', 'wlwmanifest_link');
        remove_action('wp_head', 'rsd_link');
        remove_action('wp_head', 'wp_shortlink_wp_head');
    }

    /**
     * Add canonical URL
     */
    public function add_canonical_url() {
        $canonical = '';

        if (is_singular()) {
            $canonical = get_permalink();
        } elseif (is_home() || is_front_page()) {
            $canonical = home_url('/');
        } elseif (is_category()) {
            $canonical = get_category_link(get_queried_object_id());
        } elseif (is_tag()) {
            $canonical = get_tag_link(get_queried_object_id());
        } elseif (is_author()) {
            $canonical = get_author_posts_url(get_queried_object_id());
        } elseif (is_archive()) {
            $canonical = get_pagenum_link();
        }

        // Allow filtering
        $canonical = apply_filters('seo_aeo_canonical_url', $canonical);

        if (!empty($canonical)) {
            echo '<link rel="canonical" href="' . esc_url($canonical) . '" />' . "\n";
        }
    }

    /**
     * Add meta robots tags
     */
    public function add_meta_robots() {
        $robots = array();

        // Check for noindex/nofollow on individual posts/pages
        if (is_singular()) {
            $noindex = get_post_meta(get_the_ID(), '_seo_noindex', true);
            $nofollow = get_post_meta(get_post_meta(get_the_ID(), '_seo_nofollow', true));

            if ($noindex) {
                $robots[] = 'noindex';
            }
            if ($nofollow) {
                $robots[] = 'nofollow';
            }
        }

        // Archive pages
        if (is_archive() && !empty($this->options['noindex_archives'])) {
            $robots[] = 'noindex';
        }

        // Search results
        if (is_search() && !empty($this->options['noindex_search'])) {
            $robots[] = 'noindex';
        }

        // Pagination
        if (is_paged() && !empty($this->options['noindex_paginated'])) {
            $robots[] = 'noindex';
        }

        // Image indexing control
        if (!empty($this->options['max_image_preview'])) {
            $robots[] = 'max-image-preview:' . $this->options['max_image_preview'];
        } else {
            $robots[] = 'max-image-preview:large';
        }

        // Snippet length control (for AI/search engines)
        if (!empty($this->options['max_snippet'])) {
            $robots[] = 'max-snippet:' . $this->options['max_snippet'];
        }

        // Video preview length
        if (!empty($this->options['max_video_preview'])) {
            $robots[] = 'max-video-preview:' . $this->options['max_video_preview'];
        }

        // Allow filtering
        $robots = apply_filters('seo_aeo_meta_robots', $robots);

        if (!empty($robots)) {
            echo '<meta name="robots" content="' . esc_attr(implode(', ', $robots)) . '" />' . "\n";
        }
    }

    /**
     * Optimize image attributes
     */
    public function optimize_image_attributes($attr, $attachment) {
        // Add loading="lazy" for better performance
        if (!isset($attr['loading'])) {
            $attr['loading'] = 'lazy';
        }

        // Add decoding="async"
        if (!isset($attr['decoding'])) {
            $attr['decoding'] = 'async';
        }

        // Ensure alt text exists
        if (empty($attr['alt'])) {
            $attr['alt'] = get_post_meta($attachment->ID, '_wp_attachment_image_alt', true);
            
            // Fallback to post title
            if (empty($attr['alt'])) {
                $attr['alt'] = get_the_title($attachment->ID);
            }
        }

        return $attr;
    }

    /**
     * Add resource hints (preconnect, dns-prefetch)
     */
    public function add_resource_hints() {
        $hints = array(
            'https://fonts.googleapis.com',
            'https://fonts.gstatic.com',
        );

        // Add custom hints
        if (!empty($this->options['resource_hints'])) {
            $hints = array_merge($hints, $this->options['resource_hints']);
        }

        // Allow filtering
        $hints = apply_filters('seo_aeo_resource_hints', $hints);

        foreach ($hints as $hint) {
            echo '<link rel="dns-prefetch" href="' . esc_url($hint) . '" />' . "\n";
            echo '<link rel="preconnect" href="' . esc_url($hint) . '" crossorigin />' . "\n";
        }
    }
}
