<?php
/**
 * Sitemap Generator Class
 * Dynamically generates XML sitemaps for SEO optimization
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Sitemap_Generator {

    private $options;
    private $sitemap_cache_key = 'seo_aeo_sitemap_cache';
    private $cache_duration = 86400; // 24 hours

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        // Register sitemap routes
        add_action('init', array($this, 'register_sitemap_routes'));
        
        // Flush sitemap cache on content update
        add_action('save_post', array($this, 'flush_sitemap_cache'));
        add_action('delete_post', array($this, 'flush_sitemap_cache'));
        add_action('wp_trash_post', array($this, 'flush_sitemap_cache'));
        
        // Add sitemap to robots.txt
        add_filter('robots_txt', array($this, 'add_sitemap_to_robots'), 20, 2);
    }

    /**
     * Register sitemap URL routes
     */
    public function register_sitemap_routes() {
        add_rewrite_rule('^sitemap\.xml$', 'index.php?seo_aeo_sitemap=index', 'top');
        add_rewrite_rule('^sitemap-([^/]+)\.xml$', 'index.php?seo_aeo_sitemap=$matches[1]', 'top');
        
        add_filter('query_vars', array($this, 'add_query_vars'));
        add_action('template_redirect', array($this, 'handle_sitemap_request'));
        
        // Flush rewrite rules on activation (done once)
        if (get_option('seo_aeo_sitemap_rules_flushed') !== 'yes') {
            flush_rewrite_rules();
            update_option('seo_aeo_sitemap_rules_flushed', 'yes');
        }
    }

    /**
     * Add custom query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'seo_aeo_sitemap';
        return $vars;
    }

    /**
     * Handle sitemap requests
     */
    public function handle_sitemap_request() {
        $sitemap_type = get_query_var('seo_aeo_sitemap');
        
        if (!$sitemap_type) {
            return;
        }

        // Check if sitemaps are enabled
        if (!isset($this->options['sitemap_enabled']) || !$this->options['sitemap_enabled']) {
            status_header(404);
            return;
        }

        // Generate appropriate sitemap
        if ($sitemap_type === 'index') {
            $this->output_sitemap_index();
        } else {
            $this->output_sitemap($sitemap_type);
        }
        exit;
    }

    /**
     * Output sitemap index
     */
    private function output_sitemap_index() {
        $cache_key = $this->sitemap_cache_key . '_index';
        $cached = get_transient($cache_key);
        
        if ($cached && !isset($_GET['refresh'])) {
            $this->output_xml($cached);
            return;
        }

        $sitemaps = array();
        $post_types = $this->get_enabled_post_types();
        
        foreach ($post_types as $post_type) {
            $count = wp_count_posts($post_type);
            if ($count->publish > 0) {
                $sitemaps[] = array(
                    'loc' => home_url("/sitemap-{$post_type}.xml"),
                    'lastmod' => $this->get_last_modified($post_type),
                );
            }
        }

        $xml = $this->generate_sitemap_index_xml($sitemaps);
        set_transient($cache_key, $xml, $this->cache_duration);
        
        $this->output_xml($xml);
    }

    /**
     * Output specific sitemap
     */
    private function output_sitemap($post_type) {
        $cache_key = $this->sitemap_cache_key . '_' . $post_type;
        $cached = get_transient($cache_key);
        
        if ($cached && !isset($_GET['refresh'])) {
            $this->output_xml($cached);
            return;
        }

        $urls = $this->get_sitemap_urls($post_type);
        $xml = $this->generate_sitemap_xml($urls);
        
        set_transient($cache_key, $xml, $this->cache_duration);
        $this->output_xml($xml);
    }

    /**
     * Get sitemap URLs for post type
     */
    private function get_sitemap_urls($post_type) {
        $urls = array();
        
        $args = array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        );

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                
                $url = array(
                    'loc' => get_permalink(),
                    'lastmod' => get_the_modified_date('c'),
                    'changefreq' => $this->get_changefreq($post_type),
                    'priority' => $this->get_priority($post_type),
                );
                
                // Add image if available
                if (has_post_thumbnail()) {
                    $url['image'] = get_the_post_thumbnail_url(get_the_ID(), 'full');
                }
                
                $urls[] = $url;
            }
            wp_reset_postdata();
        }

        return $urls;
    }

    /**
     * Generate sitemap index XML
     */
    private function generate_sitemap_index_xml($sitemaps) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . plugins_url('assets/sitemap-index.xsl', dirname(__FILE__)) . '"?>' . "\n";
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($sitemaps as $sitemap) {
            $xml .= '  <sitemap>' . "\n";
            $xml .= '    <loc>' . esc_url($sitemap['loc']) . '</loc>' . "\n";
            if (isset($sitemap['lastmod'])) {
                $xml .= '    <lastmod>' . esc_xml($sitemap['lastmod']) . '</lastmod>' . "\n";
            }
            $xml .= '  </sitemap>' . "\n";
        }

        $xml .= '</sitemapindex>';
        return $xml;
    }

    /**
     * Generate sitemap XML
     */
    private function generate_sitemap_xml($urls) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . plugins_url('assets/sitemap.xsl', dirname(__FILE__)) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"' . "\n";
        $xml .= '        xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";

        foreach ($urls as $url) {
            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . esc_url($url['loc']) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . esc_xml($url['lastmod']) . '</lastmod>' . "\n";
            $xml .= '    <changefreq>' . esc_xml($url['changefreq']) . '</changefreq>' . "\n";
            $xml .= '    <priority>' . esc_xml($url['priority']) . '</priority>' . "\n";
            
            if (isset($url['image'])) {
                $xml .= '    <image:image>' . "\n";
                $xml .= '      <image:loc>' . esc_url($url['image']) . '</image:loc>' . "\n";
                $xml .= '    </image:image>' . "\n";
            }
            
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';
        return $xml;
    }

    /**
     * Output XML with proper headers
     */
    private function output_xml($xml) {
        header('Content-Type: application/xml; charset=utf-8');
        header('X-Robots-Tag: noindex, follow');
        echo $xml;
    }

    /**
     * Get enabled post types
     */
    private function get_enabled_post_types() {
        $default_types = array('post', 'page');
        $enabled = isset($this->options['sitemap_post_types']) ? $this->options['sitemap_post_types'] : $default_types;
        
        // Add auction post type if it exists
        if (post_type_exists('auction') && in_array('auction', $enabled)) {
            $enabled[] = 'auction';
        }
        
        return array_unique($enabled);
    }

    /**
     * Get change frequency for post type
     */
    private function get_changefreq($post_type) {
        $frequencies = array(
            'post' => 'weekly',
            'page' => 'monthly',
            'auction' => 'daily',
        );
        
        return isset($frequencies[$post_type]) ? $frequencies[$post_type] : 'weekly';
    }

    /**
     * Get priority for post type
     */
    private function get_priority($post_type) {
        $priorities = array(
            'page' => '0.8',
            'post' => '0.6',
            'auction' => '0.7',
        );
        
        return isset($priorities[$post_type]) ? $priorities[$post_type] : '0.5';
    }

    /**
     * Get last modified date for post type
     */
    private function get_last_modified($post_type) {
        global $wpdb;
        
        $date = $wpdb->get_var($wpdb->prepare(
            "SELECT post_modified_gmt FROM {$wpdb->posts} 
            WHERE post_type = %s AND post_status = 'publish' 
            ORDER BY post_modified_gmt DESC LIMIT 1",
            $post_type
        ));
        
        return $date ? date('c', strtotime($date)) : date('c');
    }

    /**
     * Flush sitemap cache
     */
    public function flush_sitemap_cache() {
        global $wpdb;
        
        // Delete all sitemap cache transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE '_transient_{$this->sitemap_cache_key}%' 
            OR option_name LIKE '_transient_timeout_{$this->sitemap_cache_key}%'"
        );
        
        wp_cache_flush();
    }

    /**
     * Add sitemap to robots.txt
     */
    public function add_sitemap_to_robots($output, $public) {
        if ($public == '0' || !isset($this->options['sitemap_enabled']) || !$this->options['sitemap_enabled']) {
            return $output;
        }

        $sitemap_url = home_url('/sitemap.xml');
        
        // Check if sitemap is already in robots.txt
        if (strpos($output, $sitemap_url) === false) {
            $output .= "\nSitemap: " . $sitemap_url . "\n";
        }
        
        return $output;
    }

    /**
     * Ping search engines about sitemap update
     */
    public function ping_search_engines() {
        $sitemap_url = urlencode(home_url('/sitemap.xml'));
        
        // Ping Google
        wp_remote_get("https://www.google.com/ping?sitemap={$sitemap_url}");
        
        // Ping Bing
        wp_remote_get("https://www.bing.com/ping?sitemap={$sitemap_url}");
    }
}
