<?php
/**
 * Schema Generator Class
 * Handles all Schema.org structured data output
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Schema_Generator {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('wp_head', array($this, 'output_schema'), 5);
    }

    /**
     * Main method to output all schema markup
     */
    public function output_schema() {
        $schemas = array();

        // Organization schema (homepage only)
        if ($this->options['schema_organization'] && (is_front_page() || is_home())) {
            $schemas[] = $this->get_organization_schema();
        }

        // Product/Post schema
        if (is_singular()) {
            if ($this->options['schema_product'] && $this->is_product_post_type()) {
                $schemas[] = $this->get_product_schema();
            } elseif ($this->options['schema_article']) {
                $schemas[] = $this->get_article_schema();
            }
        }

        // Archive pages (auction catalogs, category pages, etc.)
        if (is_archive() || is_tax()) {
            $collection = $this->get_collection_page_schema();
            if ($collection) {
                $schemas[] = $collection;
            }
        }

        // Breadcrumb schema
        if ($this->options['schema_breadcrumb'] && !is_front_page()) {
            $breadcrumb = $this->get_breadcrumb_schema();
            if ($breadcrumb) {
                $schemas[] = $breadcrumb;
            }
        }

        // FAQ schema
        if ($this->options['schema_faq'] && !empty($this->options['faq_items'])) {
            $schemas[] = $this->get_faq_schema();
        }

        // Output all schemas
        foreach ($schemas as $schema) {
            if (!empty($schema)) {
                echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
            }
        }
    }

    /**
     * Generate Organization schema
     */
    private function get_organization_schema() {
        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'Organization',
            'name' => $this->options['org_name'],
            'url' => $this->options['org_url'],
        );

        if (!empty($this->options['org_description'])) {
            $schema['description'] = $this->options['org_description'];
        }

        if (!empty($this->options['org_logo'])) {
            $schema['logo'] = array(
                '@type' => 'ImageObject',
                'url' => $this->options['org_logo']
            );
        } elseif ($icon = get_site_icon_url()) {
            $schema['logo'] = array(
                '@type' => 'ImageObject',
                'url' => $icon
            );
        }

        if (!empty($this->options['org_social_links']) && is_array($this->options['org_social_links'])) {
            $schema['sameAs'] = $this->options['org_social_links'];
        }

        return $schema;
    }

    /**
     * Generate Product schema
     */
    private function get_product_schema() {
        global $post;

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'Product',
            'name' => get_the_title(),
            'description' => $this->get_description(),
            'url' => get_permalink(),
        );

        // Add image
        $image_url = get_the_post_thumbnail_url($post->ID, 'large');
        if ($image_url) {
            $schema['image'] = $image_url;
        }

        // Add SKU if available
        $sku = get_post_meta($post->ID, '_sku', true);
        if ($sku) {
            $schema['sku'] = $sku;
        }

        // Add offers (for WooCommerce or custom fields)
        $price = get_post_meta($post->ID, '_price', true);
        if ($price) {
            $schema['offers'] = array(
                '@type' => 'Offer',
                'price' => $price,
                'priceCurrency' => get_option('woocommerce_currency', 'USD'),
                'availability' => 'https://schema.org/InStock',
                'url' => get_permalink()
            );
        }

        return apply_filters('seo_aeo_product_schema', $schema, $post);
    }

    /**
     * Generate Article schema
     */
    private function get_article_schema() {
        global $post;

        // Get image with fallback
        $image_url = get_the_post_thumbnail_url($post->ID, 'large');
        if (!$image_url) {
            // Fallback to org logo or site icon
            $image_url = !empty($this->options['org_logo']) ? $this->options['org_logo'] : get_site_icon_url(512);
        }
        
        // Get publisher logo with fallback
        $publisher_logo = !empty($this->options['org_logo']) ? $this->options['org_logo'] : get_site_icon_url(512);

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'Article',
            'headline' => get_the_title(),
            'description' => $this->get_description(),
            'url' => get_permalink(),
            'datePublished' => get_the_date('c'),
            'dateModified' => get_the_modified_date('c'),
            'author' => array(
                '@type' => 'Person',
                'name' => get_the_author(),
                'url' => get_author_posts_url(get_the_author_meta('ID'))
            ),
            'publisher' => array(
                '@type' => 'Organization',
                'name' => $this->options['org_name'],
                'url' => home_url('/'),
                'logo' => array(
                    '@type' => 'ImageObject',
                    'url' => $publisher_logo
                )
            )
        );

        // Always include image (required by Google)
        if ($image_url) {
            $schema['image'] = array(
                '@type' => 'ImageObject',
                'url' => $image_url
            );
        }

        return apply_filters('seo_aeo_article_schema', $schema, $post);
    }

    /**
     * Generate Breadcrumb schema
     */
    private function get_breadcrumb_schema() {
        $items = array();
        $position = 1;

        // Home
        $items[] = array(
            '@type' => 'ListItem',
            'position' => $position++,
            'name' => 'Home',
            'item' => home_url('/')
        );

        // Category/Archive
        if (is_singular()) {
            global $post;
            $post_type = get_post_type();
            
            if ($post_type === 'post') {
                $categories = get_the_category();
                if (!empty($categories)) {
                    $category = $categories[0];
                    $items[] = array(
                        '@type' => 'ListItem',
                        'position' => $position++,
                        'name' => $category->name,
                        'item' => get_category_link($category->term_id)
                    );
                }
            } elseif ($post_type !== 'page') {
                $post_type_obj = get_post_type_object($post_type);
                if ($post_type_obj && $post_type_obj->has_archive) {
                    $items[] = array(
                        '@type' => 'ListItem',
                        'position' => $position++,
                        'name' => $post_type_obj->labels->name,
                        'item' => get_post_type_archive_link($post_type)
                    );
                }
            }

            // Current page
            $items[] = array(
                '@type' => 'ListItem',
                'position' => $position,
                'name' => get_the_title(),
                'item' => get_permalink()
            );
        }

        if (count($items) < 2) {
            return null;
        }

        return array(
            '@context' => 'https://schema.org/',
            '@type' => 'BreadcrumbList',
            'itemListElement' => $items
        );
    }

    /**
     * Generate FAQ schema
     */
    private function get_faq_schema() {
        if (empty($this->options['faq_items'])) {
            return null;
        }

        $questions = array();
        foreach ($this->options['faq_items'] as $faq) {
            if (!empty($faq['question']) && !empty($faq['answer'])) {
                $questions[] = array(
                    '@type' => 'Question',
                    'name' => $faq['question'],
                    'acceptedAnswer' => array(
                        '@type' => 'Answer',
                        'text' => $faq['answer']
                    )
                );
            }
        }

        if (empty($questions)) {
            return null;
        }

        return array(
            '@context' => 'https://schema.org/',
            '@type' => 'FAQPage',
            'mainEntity' => $questions
        );
    }

    /**
     * Generate CollectionPage/ItemList schema for archives
     */
    private function get_collection_page_schema() {
        global $wp_query;
        
        $archive_title = '';
        $archive_description = '';
        $archive_url = '';
        
        if (is_tax('auction_number')) {
            // Auction catalog page
            $term = get_queried_object();
            $archive_title = $term->name;
            $archive_description = $term->description;
            $archive_url = get_term_link($term);
        } elseif (is_category() || is_tag() || is_tax()) {
            // Category/Tag/Taxonomy archive
            $term = get_queried_object();
            $archive_title = $term->name;
            $archive_description = $term->description;
            $archive_url = get_term_link($term);
        } elseif (is_post_type_archive()) {
            // Post type archive
            $post_type_obj = get_queried_object();
            $archive_title = $post_type_obj->labels->name;
            $archive_url = get_post_type_archive_link($post_type_obj->name);
        } elseif (is_author()) {
            // Author archive
            $author = get_queried_object();
            $archive_title = $author->display_name;
            $archive_url = get_author_posts_url($author->ID);
        } elseif (is_date()) {
            // Date archive
            $archive_title = get_the_archive_title();
            $archive_url = get_pagenum_link();
        }
        
        if (empty($archive_title)) {
            return null;
        }
        
        // Get posts in this archive
        $items = array();
        if (have_posts()) {
            while (have_posts()) {
                the_post();
                $post_id = get_the_ID();
                
                $item = array(
                    '@type' => 'ListItem',
                    'position' => count($items) + 1,
                    'url' => get_permalink($post_id),
                    'name' => get_the_title($post_id)
                );
                
                // Add image if available
                $image_url = get_the_post_thumbnail_url($post_id, 'medium');
                if ($image_url) {
                    $item['image'] = $image_url;
                }
                
                $items[] = $item;
            }
            wp_reset_postdata();
        }
        
        if (empty($items)) {
            return null;
        }
        
        // Use CollectionPage for better semantic meaning
        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'CollectionPage',
            'name' => $archive_title,
            'url' => $archive_url,
            'mainEntity' => array(
                '@type' => 'ItemList',
                'numberOfItems' => count($items),
                'itemListElement' => $items
            )
        );
        
        if (!empty($archive_description)) {
            $schema['description'] = wp_strip_all_tags($archive_description);
        }
        
        return $schema;
    }

    /**
     * Get page/post description
     */
    private function get_description() {
        global $post;
        
        if (has_excerpt()) {
            return get_the_excerpt();
        }
        
        $content = wp_strip_all_tags($post->post_content);
        return wp_trim_words($content, 30, '...');
    }

    /**
     * Check if current post type should have product schema
     */
    private function is_product_post_type() {
        $post_type = get_post_type();
        $product_types = array('product', 'lot'); // Add more as needed
        return in_array($post_type, apply_filters('seo_aeo_product_post_types', $product_types));
    }
}
