<?php
/**
 * Meta Tags Class
 * Handles Open Graph and Twitter Card meta tags
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Meta_Tags {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('wp_head', array($this, 'output_meta_tags'), 3);
    }

    /**
     * Output all meta tags
     */
    public function output_meta_tags() {
        // Skip if Yoast or other SEO plugins are active
        if (defined('WPSEO_VERSION') || class_exists('RankMath')) {
            return;
        }

        echo "\n<!-- SEO & AEO Optimizer Meta Tags -->\n";

        if ($this->options['og_tags']) {
            $this->output_open_graph_tags();
        }

        if ($this->options['twitter_cards']) {
            $this->output_twitter_cards();
        }

        echo "<!-- / SEO & AEO Optimizer -->\n\n";
    }

    /**
     * Output Open Graph tags
     */
    private function output_open_graph_tags() {
        $og_data = $this->get_page_data();

        echo '<meta property="og:type" content="' . esc_attr($og_data['type']) . '" />' . "\n";
        echo '<meta property="og:title" content="' . esc_attr($og_data['title']) . '" />' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($og_data['description']) . '" />' . "\n";
        echo '<meta property="og:url" content="' . esc_url($og_data['url']) . '" />' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '" />' . "\n";

        // Always output og:image (required by Google, even if inferred)
        // This ensures explicit declaration as recommended
        if (!empty($og_data['image'])) {
            echo '<meta property="og:image" content="' . esc_url($og_data['image']) . '" />' . "\n";
            echo '<meta property="og:image:alt" content="' . esc_attr($og_data['title']) . '" />' . "\n";
            echo '<meta property="og:image:width" content="1200" />' . "\n";
            echo '<meta property="og:image:height" content="630" />' . "\n";
            echo '<meta property="og:image:type" content="image/jpeg" />' . "\n";
        }

        // Add locale
        echo '<meta property="og:locale" content="' . esc_attr(get_locale()) . '" />' . "\n";
    }

    /**
     * Output Twitter Card tags
     */
    private function output_twitter_cards() {
        $twitter_data = $this->get_page_data();

        echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($twitter_data['title']) . '" />' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr($twitter_data['description']) . '" />' . "\n";

        if (!empty($twitter_data['image'])) {
            echo '<meta name="twitter:image" content="' . esc_url($twitter_data['image']) . '" />' . "\n";
        }

        // Add Twitter username if set
        $twitter_username = get_option('seo_aeo_twitter_username', '');
        if (!empty($twitter_username)) {
            echo '<meta name="twitter:site" content="@' . esc_attr($twitter_username) . '" />' . "\n";
        }
    }

    /**
     * Get page data for meta tags
     */
    private function get_page_data() {
        global $post;

        $data = array(
            'type' => 'website',
            'title' => get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'url' => home_url('/'),
            'image' => ''
        );

        if (is_singular()) {
            $data['type'] = $this->get_og_type();
            $data['title'] = get_the_title();
            $data['description'] = $this->get_description();
            $data['url'] = get_permalink();
            $data['image'] = get_the_post_thumbnail_url($post->ID, 'large');
        } elseif (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            $data['title'] = $term->name;
            $data['description'] = $term->description ?: $data['description'];
            $data['url'] = get_term_link($term);
        } elseif (is_author()) {
            $author = get_queried_object();
            $data['type'] = 'profile';
            $data['title'] = $author->display_name;
            $data['description'] = $author->description ?: $data['description'];
            $data['url'] = get_author_posts_url($author->ID);
        } elseif (is_archive()) {
            $data['title'] = get_the_archive_title();
            $data['description'] = get_the_archive_description() ?: $data['description'];
            $data['url'] = get_pagenum_link();
        } elseif (is_search()) {
            $data['title'] = 'Search Results for: ' . get_search_query();
            $data['url'] = get_search_link();
        }

        // Ensure image always has a value (required by Google)
        if (empty($data['image'])) {
            // Try org logo first
            if (!empty($this->options['org_logo'])) {
                $data['image'] = $this->options['org_logo'];
            } 
            // Then site icon
            elseif ($site_icon = get_site_icon_url(512)) {
                $data['image'] = $site_icon;
            }
            // Last resort: use a data URI or default WordPress logo
            else {
                // Use WordPress default logo URL (hosted externally)
                $data['image'] = 'https://s.w.org/images/wmark.png';
            }
        }

        return apply_filters('seo_aeo_meta_data', $data);
    }

    /**
     * Get Open Graph type based on post type
     */
    private function get_og_type() {
        $post_type = get_post_type();

        $types = array(
            'product' => 'product',
            'lot' => 'product',
            'post' => 'article',
            'page' => 'website'
        );

        return isset($types[$post_type]) ? $types[$post_type] : 'website';
    }

    /**
     * Get page/post description
     */
    private function get_description() {
        global $post;

        if (has_excerpt()) {
            return wp_strip_all_tags(get_the_excerpt());
        }

        $content = wp_strip_all_tags($post->post_content);
        return wp_trim_words($content, 25, '...');
    }
}
