<?php
/**
 * Local SEO / GEO Class
 * Handles LocalBusiness schema and geographic optimization
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Local_SEO {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        if (!empty($this->options['local_seo_enabled'])) {
            add_action('wp_head', array($this, 'output_local_business_schema'), 5);
        }
    }

    /**
     * Output LocalBusiness schema for GEO/Local SEO
     */
    public function output_local_business_schema() {
        // Only on homepage or contact page
        if (!is_front_page() && !is_page($this->options['local_contact_page'] ?? 0)) {
            return;
        }

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => $this->options['local_business_type'] ?? 'LocalBusiness',
            'name' => $this->options['org_name'],
            'url' => home_url('/'),
        );

        // Add description
        if (!empty($this->options['org_description'])) {
            $schema['description'] = $this->options['org_description'];
        }

        // Add logo
        if (!empty($this->options['org_logo'])) {
            $schema['image'] = $this->options['org_logo'];
            $schema['logo'] = array(
                '@type' => 'ImageObject',
                'url' => $this->options['org_logo']
            );
        }

        // Add address (NAP - Name, Address, Phone)
        if (!empty($this->options['local_address'])) {
            $schema['address'] = array(
                '@type' => 'PostalAddress',
                'streetAddress' => $this->options['local_street_address'] ?? '',
                'addressLocality' => $this->options['local_city'] ?? '',
                'addressRegion' => $this->options['local_state'] ?? '',
                'postalCode' => $this->options['local_postal_code'] ?? '',
                'addressCountry' => $this->options['local_country'] ?? ''
            );
        }

        // Add geographic coordinates
        if (!empty($this->options['local_latitude']) && !empty($this->options['local_longitude'])) {
            $schema['geo'] = array(
                '@type' => 'GeoCoordinates',
                'latitude' => $this->options['local_latitude'],
                'longitude' => $this->options['local_longitude']
            );
        }

        // Add contact information
        if (!empty($this->options['local_phone'])) {
            $schema['telephone'] = $this->options['local_phone'];
        }

        if (!empty($this->options['local_email'])) {
            $schema['email'] = $this->options['local_email'];
        }

        // Add opening hours
        if (!empty($this->options['local_opening_hours'])) {
            $schema['openingHours'] = $this->options['local_opening_hours'];
        }

        // Add price range
        if (!empty($this->options['local_price_range'])) {
            $schema['priceRange'] = $this->options['local_price_range'];
        }

        // Add social media
        if (!empty($this->options['org_social_links'])) {
            $schema['sameAs'] = $this->options['org_social_links'];
        }

        // Add service area (for businesses serving multiple areas)
        if (!empty($this->options['local_service_areas'])) {
            $areas = array();
            foreach ($this->options['local_service_areas'] as $area) {
                $areas[] = array(
                    '@type' => 'City',
                    'name' => $area
                );
            }
            $schema['areaServed'] = $areas;
        }

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }
}
