<?php
/**
 * Auction Schema Extension
 * Adds support for UCO Auction sites
 */

if (!defined('ABSPATH')) {
    exit;
}

// Only load if auction plugin is active
if (!class_exists('uco_bidspirit_auction_integration\inc\helpers\LotHelper')) {
    return;
}

// Load compatibility layer
require_once SEO_AEO_PLUGIN_DIR . 'includes/class-auction-compatibility.php';

use uco_bidspirit_auction_integration\inc\helpers\LotHelper;
use uco_bidspirit_auction_integration\inc\helpers\SettingsHelper;
use uco_bidspirit_auction_integration\inc\model\Auction;

class SEO_AEO_Auction_Schema {

    public function __construct() {
        // Hook into plugin's product schema filter
        add_filter('seo_aeo_product_schema', array($this, 'customize_lot_schema'), 10, 2);
        add_filter('seo_aeo_product_post_types', array($this, 'add_lot_post_type'));
        
        // Add auction-specific schemas
        add_action('wp_head', array($this, 'add_auction_event_schema'), 8);
        add_action('wp_head', array($this, 'add_auction_breadcrumbs'), 9);
        
        // Enhance meta descriptions for lots
        add_filter('seo_aeo_meta_data', array($this, 'enhance_lot_meta'), 10);
    }

    /**
     * Add 'lot' to product post types
     */
    public function add_lot_post_type($types) {
        $types[] = 'lot';
        return $types;
    }

    /**
     * Customize schema for auction lots (with fallbacks)
     */
    public function customize_lot_schema($schema, $post) {
        if (get_post_type($post) !== 'lot') {
            return $schema;
        }

        // Use compatibility layer to safely get lot data
        $lotData = SEO_AEO_Auction_Compatibility::get_lot_data($post->ID);
        
        // Enhance schema with auction data
        $schema['sku'] = 'LOT-' . $lotData['lot_index'];
        $schema['productID'] = $lotData['lot_index'];
        
        if (!empty($lotData['author'])) {
            $schema['creator'] = array(
                '@type' => 'Person',
                'name' => $lotData['author']
            );
        }
        
        if ($lotData['start_price'] > 0) {
            $schema['offers']['price'] = $lotData['start_price'];
            $schema['offers']['priceCurrency'] = 'ILS';
        }
        
        // Add auction date if available
        if (!empty($lotData['auction_date'])) {
            $schema['offers']['priceValidUntil'] = date('Y-m-d', strtotime($lotData['auction_date']));
        }
        
        return $schema;
    }

    /**
     * Add SaleEvent schema for auction lots (with fallbacks)
     */
    public function add_auction_event_schema() {
        if (!is_singular('lot')) {
            return;
        }

        global $post;
        
        // Use compatibility layer
        $lotData = SEO_AEO_Auction_Compatibility::get_lot_data($post->ID);
        
        if (empty($lotData['auction_date'])) {
            return; // No date available
        }
        
        $auctionName = $lotData['auction_name'];
        $lotIndex = $lotData['lot_index'];
        $title = $lotData['title'];
        $dateTime = $lotData['auction_date'];
        $startPrice = $lotData['start_price'];
        
        $saleEventSchema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'SaleEvent',
            'name' => $auctionName . ' - Lot ' . $lotIndex,
            'startDate' => date('c', strtotime($dateTime)),
            'location' => array(
                '@type' => 'VirtualLocation',
                'url' => get_permalink($post->ID)
            ),
            'organizer' => array(
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                'url' => home_url('/')
            ),
            'about' => array(
                '@type' => 'Product',
                'name' => $title,
            )
        );
        
        if ($startPrice > 0) {
            $saleEventSchema['offers'] = array(
                '@type' => 'Offer',
                'price' => $startPrice,
                'priceCurrency' => 'ILS',
                'url' => get_permalink($post->ID)
            );
        }
        
        echo '<script type="application/ld+json">' . wp_json_encode($saleEventSchema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Add auction-specific breadcrumb schema
     */
    public function add_auction_breadcrumbs() {
        if (!is_singular('lot')) {
            return;
        }

        global $post;
        $auction = get_the_terms($post->ID, 'auction_number');
        
        if (empty($auction)) {
            return;
        }
        
        $auction = $auction[0];
        
        $breadcrumbs = array(
            '@context' => 'https://schema.org/',
            '@type' => 'BreadcrumbList',
            'itemListElement' => array(
                array(
                    '@type' => 'ListItem',
                    'position' => 1,
                    'name' => 'Home',
                    'item' => home_url('/')
                ),
                array(
                    '@type' => 'ListItem',
                    'position' => 2,
                    'name' => $auction->name,
                    'item' => get_term_link($auction)
                ),
                array(
                    '@type' => 'ListItem',
                    'position' => 3,
                    'name' => get_the_title(),
                    'item' => get_permalink($post->ID)
                )
            )
        );
        
        echo '<script type="application/ld+json">' . wp_json_encode($breadcrumbs, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Enhance meta description for auction lots (with fallbacks)
     */
    public function enhance_lot_meta($data) {
        if (!is_singular('lot') || !isset($data['description'])) {
            return $data;
        }

        global $post;
        
        // Use compatibility layer
        $lotData = SEO_AEO_Auction_Compatibility::get_lot_data($post->ID);
        
        $lotIndex = $lotData['lot_index'];
        $title = $lotData['title'];
        $author = $lotData['author'];
        $estimatedPrice = $lotData['estimated_price'];
        
        // Build AI-friendly description
        $enhanced = "Lot " . $lotIndex . ": " . $title;
        
        if (!empty($author)) {
            $enhanced .= " by " . $author;
        }
        
        if (!empty($estimatedPrice)) {
            $enhanced .= ". Estimated price: " . $estimatedPrice;
        }
        
        $content = wp_strip_all_tags(get_the_content());
        if (!empty($content)) {
            $enhanced .= ". " . wp_trim_words($content, 20, '...');
        }
        
        $enhanced .= " Bid online now.";
        
        $data['description'] = $enhanced;
        $data['title'] = 'Lot ' . $lotIndex . ': ' . $title;
        
        return $data;
    }
}

// Initialize if auction plugin is active
if (class_exists('uco_bidspirit_auction_integration\inc\helpers\LotHelper')) {
    new SEO_AEO_Auction_Schema();
}
