<?php
/**
 * Auction Plugin Compatibility Layer
 * Ensures SEO/AEO/GEO continues working even when UCO auctions plugin updates
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Auction_Compatibility {

    private $uco_plugin_version = null;
    private $compatibility_mode = 'direct'; // 'direct', 'fallback', or 'disabled'

    public function __construct() {
        // Check on admin page loads
        add_action('admin_init', array($this, 'check_compatibility'));
        
        // Show admin notices
        add_action('admin_notices', array($this, 'compatibility_notices'));
        
        // Schedule daily automatic check
        add_action('wp', array($this, 'schedule_daily_check'));
        add_action('seo_aeo_daily_compat_check', array($this, 'check_compatibility'));
        
        // Immediate check when UCO plugin updates
        add_action('upgrader_process_complete', array($this, 'check_after_plugin_update'), 10, 2);
        
        // Check when any plugin is activated/deactivated
        add_action('activated_plugin', array($this, 'check_on_plugin_change'));
        add_action('deactivated_plugin', array($this, 'check_on_plugin_change'));
    }

    /**
     * Check UCO plugin compatibility
     */
    public function check_compatibility() {
        // Update last check timestamp
        update_option('seo_aeo_last_compat_check', current_time('mysql'));
        
        // Detect UCO plugin version
        $this->uco_plugin_version = $this->detect_uco_version();
        
        // Test if critical classes exist
        $has_lot_helper = class_exists('uco_bidspirit_auction_integration\inc\helpers\LotHelper');
        $has_settings_helper = class_exists('uco_bidspirit_auction_integration\inc\helpers\SettingsHelper');
        $has_auction_model = class_exists('uco_bidspirit_auction_integration\inc\model\Auction');
        
        // Determine compatibility mode
        if ($has_lot_helper && $has_settings_helper && $has_auction_model) {
            $this->compatibility_mode = 'direct';
            update_option('seo_aeo_auction_compat_mode', 'direct');
            update_option('seo_aeo_auction_compat_version', $this->uco_plugin_version);
        } else {
            $this->compatibility_mode = 'fallback';
            update_option('seo_aeo_auction_compat_mode', 'fallback');
            
            // Log the issue
            $this->log_compatibility_issue('UCO plugin classes not found or changed');
        }
    }

    /**
     * Schedule daily compatibility check
     */
    public function schedule_daily_check() {
        if (!wp_next_scheduled('seo_aeo_daily_compat_check')) {
            wp_schedule_event(time(), 'daily', 'seo_aeo_daily_compat_check');
        }
    }

    /**
     * Check compatibility after plugin update
     */
    public function check_after_plugin_update($upgrader_object, $options) {
        // Only check if plugins were updated
        if ($options['action'] !== 'update' || $options['type'] !== 'plugin') {
            return;
        }

        // Check if UCO plugin was updated
        $updated_plugins = isset($options['plugins']) ? $options['plugins'] : array();
        
        foreach ($updated_plugins as $plugin) {
            if (strpos($plugin, 'uco-auctions') !== false) {
                // UCO plugin was updated - run immediate compatibility check
                $this->check_compatibility();
                
                // Log the update
                $this->log_compatibility_issue('UCO Auctions plugin was updated - compatibility re-validated');
                break;
            }
        }
    }

    /**
     * Check on plugin activation/deactivation
     */
    public function check_on_plugin_change($plugin) {
        if (strpos($plugin, 'uco-auctions') !== false) {
            $this->check_compatibility();
        }
    }

    /**
     * Detect UCO plugin version
     */
    private function detect_uco_version() {
        $plugin_file = WP_PLUGIN_DIR . '/uco-auctions/uco-auctions.php';
        
        if (!file_exists($plugin_file)) {
            return 'not-installed';
        }
        
        $plugin_data = get_plugin_data($plugin_file, false, false);
        return $plugin_data['Version'] ?? 'unknown';
    }

    /**
     * Show admin notices for compatibility issues
     */
    public function compatibility_notices() {
        $mode = get_option('seo_aeo_auction_compat_mode', 'direct');
        
        if ($mode === 'fallback') {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p><strong>SEO & AEO Optimizer:</strong> UCO Auctions plugin structure has changed. Using fallback mode for SEO. Some advanced features may be limited.</p>
                <p><a href="<?php echo admin_url('admin.php?page=seo-aeo-optimizer&tab=tools'); ?>">Check Compatibility Status</a></p>
            </div>
            <?php
        }
        
        // Check for version changes
        $last_version = get_option('seo_aeo_auction_compat_version', 'unknown');
        $current_version = $this->detect_uco_version();
        
        if ($last_version !== $current_version && $last_version !== 'unknown') {
            ?>
            <div class="notice notice-info is-dismissible">
                <p><strong>SEO & AEO Optimizer:</strong> UCO Auctions plugin was updated from v<?php echo esc_html($last_version); ?> to v<?php echo esc_html($current_version); ?>. SEO features have been re-validated.</p>
            </div>
            <?php
            update_option('seo_aeo_auction_compat_version', $current_version);
        }
    }

    /**
     * Log compatibility issues
     */
    private function log_compatibility_issue($message) {
        $log = get_option('seo_aeo_auction_compat_log', array());
        $log[] = array(
            'time' => current_time('mysql'),
            'message' => $message,
            'version' => $this->uco_plugin_version
        );
        
        // Keep only last 10 entries
        if (count($log) > 10) {
            $log = array_slice($log, -10);
        }
        
        update_option('seo_aeo_auction_compat_log', $log);
    }

    /**
     * Safe method to get lot data with fallbacks
     */
    public static function get_lot_data($post_id) {
        $data = array(
            'lot_index' => 0,
            'title' => get_the_title($post_id),
            'author' => '',
            'start_price' => 0,
            'estimated_price' => '',
            'auction_date' => '',
            'auction_name' => ''
        );

        try {
            // Try direct method first
            if (class_exists('uco_bidspirit_auction_integration\inc\helpers\LotHelper')) {
                $lotHelper = new \uco_bidspirit_auction_integration\inc\helpers\LotHelper($post_id);
                
                $data['lot_index'] = $lotHelper->getOptionValue('lot_itemIndex', 0);
                $data['title'] = $lotHelper->getOptionValue('name', get_the_title($post_id));
                $data['author'] = $lotHelper->getOptionValue('lot_author', '');
                $data['start_price'] = $lotHelper->getOptionValue('lot_start_price', 0);
                $data['estimated_price'] = $lotHelper->getOptionValue('lot_estimated_price', '');
            } else {
                // Fallback to post meta
                $data['lot_index'] = get_post_meta($post_id, 'lot_itemIndex', true) ?: 0;
                $data['author'] = get_post_meta($post_id, 'lot_author', true) ?: '';
                $data['start_price'] = get_post_meta($post_id, 'lot_start_price', true) ?: 0;
                $data['estimated_price'] = get_post_meta($post_id, 'lot_estimated_price', true) ?: '';
            }

            // Get auction info
            $auction_terms = get_the_terms($post_id, 'auction_number');
            if (!empty($auction_terms)) {
                $auction = $auction_terms[0];
                $data['auction_name'] = $auction->name;
                
                // Try to get auction date
                if (class_exists('uco_bidspirit_auction_integration\inc\helpers\SettingsHelper') && 
                    class_exists('uco_bidspirit_auction_integration\inc\model\Auction')) {
                    
                    $auctionDays = \uco_bidspirit_auction_integration\inc\helpers\SettingsHelper::getTermOption($auction->term_id, 'auction_days');
                    $auctionDay = \uco_bidspirit_auction_integration\inc\model\Auction::getAuctionDayByItemID($data['lot_index'], $auctionDays);
                    
                    if (isset($auctionDay->date)) {
                        $data['auction_date'] = !empty($auctionDay->unknownExactDate) ? 
                            $auctionDay->date : 
                            $auctionDay->date . 'T' . $auctionDay->time;
                    }
                } else {
                    // Fallback: try to get date from term meta
                    $data['auction_date'] = get_term_meta($auction->term_id, 'auction_date', true);
                }
            }

        } catch (Exception $e) {
            // Log error but continue with defaults
            error_log('SEO AEO Optimizer: Error getting lot data - ' . $e->getMessage());
        }

        return $data;
    }

    /**
     * Check if method exists on object (safe way)
     */
    public static function safe_method_call($object, $method, $default = null) {
        if (is_object($object) && method_exists($object, $method)) {
            try {
                return call_user_func(array($object, $method));
            } catch (Exception $e) {
                error_log('SEO AEO Optimizer: Method call failed - ' . $method);
                return $default;
            }
        }
        return $default;
    }

    /**
     * Get compatibility status for display
     */
    public static function get_status() {
        $mode = get_option('seo_aeo_auction_compat_mode', 'unknown');
        $version = get_option('seo_aeo_auction_compat_version', 'unknown');
        $log = get_option('seo_aeo_auction_compat_log', array());

        return array(
            'mode' => $mode,
            'version' => $version,
            'log' => $log,
            'classes_available' => array(
                'LotHelper' => class_exists('uco_bidspirit_auction_integration\inc\helpers\LotHelper'),
                'SettingsHelper' => class_exists('uco_bidspirit_auction_integration\inc\helpers\SettingsHelper'),
                'Auction' => class_exists('uco_bidspirit_auction_integration\inc\model\Auction')
            )
        );
    }
}

// Initialize compatibility checker
new SEO_AEO_Auction_Compatibility();
