<?php
/**
 * Advanced Schema Types
 * Reviews, Events, HowTo, Speakable, Video, etc.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Advanced_Schema {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('wp_head', array($this, 'output_advanced_schemas'), 8);
    }

    public function output_advanced_schemas() {
        // Event schema (for auction events)
        if ($this->options['schema_event'] ?? false) {
            $this->add_event_schema();
        }

        // Review/Rating schema
        if ($this->options['schema_review'] ?? false) {
            $this->add_review_schema();
        }

        // Video schema
        if ($this->options['schema_video'] ?? false) {
            $this->add_video_schema();
        }

        // HowTo schema (for AEO)
        if ($this->options['schema_howto'] ?? false) {
            $this->add_howto_schema();
        }

        // Speakable schema (voice search optimization)
        if ($this->options['schema_speakable'] ?? false) {
            $this->add_speakable_schema();
        }
    }

    /**
     * Event Schema - Perfect for Auctions
     */
    private function add_event_schema() {
        if (!is_singular()) {
            return;
        }

        global $post;
        
        // Check if post has event meta
        $event_date = get_post_meta($post->ID, '_event_date', true);
        $event_end_date = get_post_meta($post->ID, '_event_end_date', true);
        $event_location = get_post_meta($post->ID, '_event_location', true);
        
        if (empty($event_date)) {
            return;
        }

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'Event',
            'name' => get_the_title(),
            'description' => $this->get_description(),
            'startDate' => date('c', strtotime($event_date)),
            'url' => get_permalink()
        );

        if (!empty($event_end_date)) {
            $schema['endDate'] = date('c', strtotime($event_end_date));
        }

        // Add location
        if (!empty($event_location)) {
            $schema['location'] = array(
                '@type' => 'Place',
                'name' => $event_location,
                'address' => $event_location
            );
        } else {
            // Virtual event
            $schema['location'] = array(
                '@type' => 'VirtualLocation',
                'url' => get_permalink()
            );
        }

        // Add organizer
        $schema['organizer'] = array(
            '@type' => 'Organization',
            'name' => $this->options['org_name'],
            'url' => home_url('/')
        );

        // Add image
        $image = get_the_post_thumbnail_url($post->ID, 'large');
        if ($image) {
            $schema['image'] = $image;
        }

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Review/Rating Schema
     */
    private function add_review_schema() {
        if (!is_singular()) {
            return;
        }

        global $post;

        $rating = get_post_meta($post->ID, '_rating', true);
        $review_count = get_post_meta($post->ID, '_review_count', true);
        
        if (empty($rating)) {
            return;
        }

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'Product',
            'name' => get_the_title(),
            'aggregateRating' => array(
                '@type' => 'AggregateRating',
                'ratingValue' => $rating,
                'ratingCount' => $review_count ?? 1,
                'bestRating' => '5',
                'worstRating' => '1'
            )
        );

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Video Schema
     */
    private function add_video_schema() {
        if (!is_singular()) {
            return;
        }

        global $post;

        $video_url = get_post_meta($post->ID, '_video_url', true);
        $video_thumbnail = get_post_meta($post->ID, '_video_thumbnail', true);
        $video_duration = get_post_meta($post->ID, '_video_duration', true);
        
        if (empty($video_url)) {
            return;
        }

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'VideoObject',
            'name' => get_the_title(),
            'description' => $this->get_description(),
            'contentUrl' => $video_url,
            'uploadDate' => get_the_date('c'),
            'thumbnailUrl' => $video_thumbnail ?? get_the_post_thumbnail_url($post->ID, 'large')
        );

        if (!empty($video_duration)) {
            $schema['duration'] = $video_duration;
        }

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * HowTo Schema - Perfect for AEO (AI Answer Engines)
     */
    private function add_howto_schema() {
        // Only add on pages/posts with HowTo content
        if (!is_singular() || !has_block('core/heading')) {
            return;
        }

        global $post;

        // Parse content for steps (H2, H3 headings or numbered lists)
        $content = $post->post_content;
        preg_match_all('/<h[23][^>]*>(.*?)<\/h[23]>/i', $content, $headings);
        
        if (empty($headings[1]) || count($headings[1]) < 2) {
            return;
        }

        $steps = array();
        foreach ($headings[1] as $index => $heading) {
            $steps[] = array(
                '@type' => 'HowToStep',
                'position' => $index + 1,
                'name' => strip_tags($heading),
                'text' => strip_tags($heading)
            );
        }

        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'HowTo',
            'name' => get_the_title(),
            'description' => $this->get_description(),
            'step' => $steps
        );

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Speakable Schema - Voice Search Optimization (AEO)
     */
    private function add_speakable_schema() {
        // Only add on singular pages and homepage
        if (!is_singular() && !is_front_page()) {
            return;
        }

        // Use more flexible selectors that match common WordPress structures
        $selectors = array();
        
        if (is_front_page()) {
            // Homepage selectors - broader for various homepage layouts
            $selectors = array('h1', 'h2', '.site-description', '.page-content p', 'main p');
        } else {
            // Single post/page selectors
            $selectors = array('h1', 'h2', '.entry-content p', '.content p', 'article p');
        }

        // Mark key sections as speakable for voice assistants
        $schema = array(
            '@context' => 'https://schema.org/',
            '@type' => 'WebPage',
            'name' => is_front_page() ? get_bloginfo('name') : get_the_title(),
            'speakable' => array(
                '@type' => 'SpeakableSpecification',
                'cssSelector' => $selectors
            )
        );

        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>' . "\n";
    }

    /**
     * Get description helper
     */
    private function get_description() {
        global $post;
        
        if (has_excerpt()) {
            return wp_strip_all_tags(get_the_excerpt());
        }
        
        $content = wp_strip_all_tags($post->post_content);
        return wp_trim_words($content, 30, '...');
    }
}
