<?php
/**
 * Admin Settings Panel
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_AEO_Admin_Settings {

    private $options;

    public function __construct($options) {
        $this->options = $options;
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    public function add_admin_menu() {
        add_menu_page(
            __('SEO & AEO Optimizer', 'seo-aeo-optimizer'),
            __('SEO & AEO', 'seo-aeo-optimizer'),
            'manage_options',
            'seo-aeo-optimizer',
            array($this, 'render_settings_page'),
            'dashicons-search',
            80
        );
    }

    public function register_settings() {
        register_setting('seo_aeo_options_group', 'seo_aeo_options', array($this, 'sanitize_options'));
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_seo-aeo-optimizer') {
            return;
        }
        
        wp_enqueue_style('seo-aeo-admin', SEO_AEO_PLUGIN_URL . 'assets/css/admin.css', array(), SEO_AEO_VERSION);
        wp_enqueue_script('seo-aeo-admin', SEO_AEO_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), SEO_AEO_VERSION, true);
    }

    public function sanitize_options($input) {
        // Get existing options to preserve values from other tabs
        $existing = get_option('seo_aeo_options', array());
        $sanitized = $existing; // Start with existing values
        
        // Group checkboxes by tab to handle partial saves properly
        $checkbox_groups = array(
            'advanced' => array('schema_event', 'schema_review', 'schema_video', 'schema_howto', 'schema_speakable',
                               'canonical_enabled', 'image_optimization', 'dns_prefetch',
                               'noindex_archives', 'noindex_search', 'noindex_paginated'),
            'sitemap' => array('sitemap_enabled'),
            'local' => array('local_seo_enabled'),
            'general' => array('enabled', 'schema', 'debug_mode', 'cache_schema'),
            'schema' => array('schema_organization', 'schema_product', 'schema_breadcrumb', 'schema_faq', 'schema_article'),
            'meta' => array('meta_tags', 'og_tags', 'twitter_cards'),
            'ai' => array('robots', 'allow_gptbot', 'allow_ccbot', 'allow_anthropic', 'allow_perplexity', 'allow_youbot'),
            'faq' => array('faq_enabled')
        );
        
        // Determine which group is being saved
        $active_groups = array();
        
        // Check for hidden tab identifier fields (most reliable)
        if (isset($input['_tab_advanced'])) {
            $active_groups[] = 'advanced';
        }
        // Fallback: check for select/textarea fields unique to each tab
        elseif (isset($input['max_image_preview']) || isset($input['max_snippet'])) {
            $active_groups[] = 'advanced';
        }
        if (isset($input['local_business_type']) || isset($input['local_city'])) {
            $active_groups[] = 'local';
        }
        if (isset($input['org_name'])) {
            $active_groups[] = 'general';
        }
        if (isset($input['faq_items'])) {
            $active_groups[] = 'faq';
        }
        
        // Update checkboxes only for active groups
        foreach ($checkbox_groups as $group => $group_checkboxes) {
            if (in_array($group, $active_groups) || !empty(array_intersect($group_checkboxes, array_keys($input)))) {
                foreach ($group_checkboxes as $checkbox) {
                    $sanitized[$checkbox] = isset($input[$checkbox]) ? true : false;
                }
            }
        }
        
        // General text fields - only update if present in input
        if (isset($input['org_name'])) $sanitized['org_name'] = sanitize_text_field($input['org_name']);
        if (isset($input['org_url'])) $sanitized['org_url'] = esc_url_raw($input['org_url']);
        if (isset($input['org_logo'])) $sanitized['org_logo'] = esc_url_raw($input['org_logo']);
        if (isset($input['org_description'])) $sanitized['org_description'] = sanitize_textarea_field($input['org_description']);
        
        // Sitemap settings - only update if present
        if (isset($input['sitemap_post_types'])) {
            $sanitized['sitemap_post_types'] = array_map('sanitize_key', $input['sitemap_post_types']);
        }
        
        // Local/GEO fields - only update if present in input
        if (isset($input['local_business_type'])) $sanitized['local_business_type'] = sanitize_text_field($input['local_business_type']);
        if (isset($input['local_street_address'])) $sanitized['local_street_address'] = sanitize_text_field($input['local_street_address']);
        if (isset($input['local_city'])) $sanitized['local_city'] = sanitize_text_field($input['local_city']);
        if (isset($input['local_state'])) $sanitized['local_state'] = sanitize_text_field($input['local_state']);
        if (isset($input['local_postal_code'])) $sanitized['local_postal_code'] = sanitize_text_field($input['local_postal_code']);
        if (isset($input['local_country'])) $sanitized['local_country'] = sanitize_text_field($input['local_country']);
        if (isset($input['local_phone'])) $sanitized['local_phone'] = sanitize_text_field($input['local_phone']);
        if (isset($input['local_email'])) $sanitized['local_email'] = sanitize_email($input['local_email']);
        if (isset($input['local_latitude'])) $sanitized['local_latitude'] = sanitize_text_field($input['local_latitude']);
        if (isset($input['local_longitude'])) $sanitized['local_longitude'] = sanitize_text_field($input['local_longitude']);
        if (isset($input['local_price_range'])) $sanitized['local_price_range'] = sanitize_text_field($input['local_price_range']);
        if (isset($input['local_opening_hours'])) $sanitized['local_opening_hours'] = sanitize_textarea_field($input['local_opening_hours']);
        if (isset($input['local_service_areas'])) $sanitized['local_service_areas'] = sanitize_textarea_field($input['local_service_areas']);
        
        // Technical SEO selects - only update if present in input
        if (isset($input['max_image_preview'])) $sanitized['max_image_preview'] = sanitize_text_field($input['max_image_preview']);
        if (isset($input['max_snippet'])) $sanitized['max_snippet'] = sanitize_text_field($input['max_snippet']);
        if (isset($input['max_video_preview'])) $sanitized['max_video_preview'] = sanitize_text_field($input['max_video_preview']);
        
        // Arrays - only update if present in input
        if (isset($input['post_types'])) {
            $sanitized['post_types'] = array_map('sanitize_text_field', $input['post_types']);
        }
        if (isset($input['org_social_links'])) {
            $sanitized['org_social_links'] = array_map('esc_url_raw', $input['org_social_links']);
        }
        
        // FAQ items - only update if present in input
        if (isset($input['faq_items']) && is_array($input['faq_items'])) {
            $sanitized['faq_items'] = array();
            foreach ($input['faq_items'] as $item) {
                if (!empty($item['question']) && !empty($item['answer'])) {
                    $sanitized['faq_items'][] = array(
                        'question' => sanitize_text_field($item['question']),
                        'answer' => sanitize_textarea_field($item['answer'])
                    );
                }
            }
        }
        
        return $sanitized;
    }

    public function render_settings_page() {
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
        ?>
        <div class="wrap seo-aeo-settings">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <h2 class="nav-tab-wrapper">
                <a href="?page=seo-aeo-optimizer&tab=general" class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('General', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=schema" class="nav-tab <?php echo $active_tab === 'schema' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Schema', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=local" class="nav-tab <?php echo $active_tab === 'local' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Local/GEO', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=advanced" class="nav-tab <?php echo $active_tab === 'advanced' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Advanced', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=meta" class="nav-tab <?php echo $active_tab === 'meta' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Meta Tags', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=ai" class="nav-tab <?php echo $active_tab === 'ai' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('AI Crawlers', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=sitemap" class="nav-tab <?php echo $active_tab === 'sitemap' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('XML Sitemap', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=faq" class="nav-tab <?php echo $active_tab === 'faq' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('FAQ Schema', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=updates" class="nav-tab <?php echo $active_tab === 'updates' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Updates', 'seo-aeo-optimizer'); ?>
                </a>
                <a href="?page=seo-aeo-optimizer&tab=tools" class="nav-tab <?php echo $active_tab === 'tools' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Tools', 'seo-aeo-optimizer'); ?>
                </a>
            </h2>

            <form method="post" action="options.php">
                <?php
                settings_fields('seo_aeo_options_group');
                $options = get_option('seo_aeo_options');
                
                switch ($active_tab) {
                    case 'schema':
                        $this->render_schema_tab($options);
                        break;
                    case 'local':
                        $this->render_local_tab($options);
                        break;
                    case 'advanced':
                        $this->render_advanced_tab($options);
                        break;
                    case 'meta':
                        $this->render_meta_tab($options);
                        break;
                    case 'ai':
                        $this->render_ai_tab($options);
                        break;
                    case 'sitemap':
                        include SEO_AEO_PLUGIN_DIR . 'admin/tabs/sitemap.php';
                        break;
                    case 'faq':
                        $this->render_faq_tab($options);
                        break;
                    case 'updates':
                        $this->render_updates_tab($options);
                        break;
                    case 'tools':
                        $this->render_tools_tab($options);
                        break;
                    default:
                        $this->render_general_tab($options);
                }
                
                if ($active_tab !== 'tools' && $active_tab !== 'updates') {
                    submit_button();
                }
                ?>
            </form>
        </div>
        <?php
    }

    private function render_general_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/general.php';
    }

    private function render_schema_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/schema.php';
    }

    private function render_local_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/local.php';
    }

    private function render_advanced_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/advanced.php';
    }

    private function render_meta_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/meta.php';
    }

    private function render_ai_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/ai.php';
    }

    private function render_faq_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/faq.php';
    }

    private function render_tools_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/tools.php';
    }
    
    private function render_updates_tab($options) {
        require_once SEO_AEO_PLUGIN_DIR . 'admin/tabs/updates.php';
    }
}
